// Copyright 2014 The Chromium OS Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "volume.h"

#include "gtest/gtest.h"
#include "ppapi_simple/ps_main.h"

#include "request.h"

namespace {

// A file system id used at the creation of Volume.
const char kFileSystemId[] = "fileSystemId";

// A fake implementation of JavaScriptMessageSender used for testing purposes.
class FakeJavaScriptMessageSender : public JavaScriptMessageSenderInterface {
 public:
  virtual void SendFileSystemError(const std::string& file_system_id,
                                   const std::string& request_id,
                                   const std::string& message) {}

  virtual void SendCompressorError(int compressor_id,
                                   const std::string& message) {};

  virtual void SendFileChunkRequest(const std::string& file_system_id,
                                    const std::string& request_id,
                                    int64_t offset,
                                    int64_t bytes_to_read) {}

  virtual void SendPassphraseRequest(const std::string& file_system_id,
                                     const std::string& request_id) {}

  virtual void SendReadMetadataDone(const std::string& file_system_id,
                                    const std::string& request_id,
                                    const pp::VarDictionary& metadata) {}

  virtual void SendOpenFileDone(const std::string& file_system_id,
                                const std::string& request_id) {}

  virtual void SendCloseFileDone(const std::string& file_system_id,
                                 const std::string& request_id,
                                 const std::string& open_request_id) {}

  virtual void SendReadFileDone(const std::string& file_system_id,
                                const std::string& request_id,
                                const pp::VarArrayBuffer& array_buffer,
                                bool has_more_data) {}

  virtual void SendConsoleLog(const std::string& file_system_id,
                              const std::string& request_id,
                              const std::string& src_file,
                              int src_line,
                              const std::string& src_func,
                              const std::string& message) {}

  virtual void SendCreateArchiveDone(int compressor_id) {};

  virtual void SendReadFileChunk(int compressor_id_,
                                 int64_t file_size) {};

  virtual void SendWriteChunk(int compressor_id,
                              const pp::VarArrayBuffer& array_buffer,
                              int64_t length) {};

  virtual void SendAddToArchiveDone(int compressor_id) {};

  virtual void SendCloseArchiveDone(int compressor_id) {};
};

}  // namespace

// Class used by TEST_F macro to initialize the environment for testing
// Volume methods.
class VolumeTest : public testing::Test {
 protected:
  VolumeTest() : message_sender(NULL), volume(NULL) {}

  virtual void SetUp() {
    message_sender = new FakeJavaScriptMessageSender();
    // TODO(cmihail): Use the constructor with custom factories for
    // VolumeArchive and VolumeReader.
    volume = new Volume(
        pp::InstanceHandle(PSGetInstanceId()), kFileSystemId, message_sender);
  }

  virtual void TearDown() {
    delete message_sender;
    message_sender = NULL;
    delete volume;
    volume = NULL;
  }

  FakeJavaScriptMessageSender* message_sender;
  Volume* volume;
};

TEST_F(VolumeTest, Init) {
  EXPECT_TRUE(volume->Init());
}

// TODO(cmihail): Write the actual tests (see crbug.com/417973).
