// Copyright 2017 The Chromium OS Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPRESSOR_H_
#define COMPRESSOR_H_

#include <ctime>
#include <pthread.h>

#include "archive.h"
#include "ppapi/cpp/instance_handle.h"
#include "ppapi/cpp/var_array_buffer.h"
#include "ppapi/cpp/var_dictionary.h"
#include "ppapi/utility/completion_callback_factory.h"
#include "ppapi/utility/threading/simple_thread.h"

#include "compressor_archive.h"
#include "compressor_stream.h"
#include "javascript_compressor_requestor_interface.h"
#include "javascript_message_sender_interface.h"

// Handles all packing operations like creating archive objects and writing data
// onto the archive.
class Compressor {
 public:
  Compressor(const pp::InstanceHandle& instance_handle /* Used for workers. */,
             int compressor_id,
             JavaScriptMessageSenderInterface* message_sender);

  virtual ~Compressor();

  // Initializes the compressor.
  bool Init();

  // Creates an archive object.
  void CreateArchive();

  // Adds an entry to the archive.
  void AddToArchive(const pp::VarDictionary& dictionary);

  // Processes a file chunk sent from JavaScript.
  void ReadFileChunkDone(const pp::VarDictionary& dictionary);

  // Receives a write chunk response from JavaScript.
  void WriteChunkDone(const pp::VarDictionary& dictionary);

  // Releases all resources obtained by libarchive.
  void CloseArchive(const pp::VarDictionary& dictionary);

  // A getter function for the message sender.
  JavaScriptMessageSenderInterface* message_sender() { return message_sender_; }

  // A getter function for the requestor.
  JavaScriptCompressorRequestorInterface* requestor() { return requestor_; }

  // A getter function for the compressor id.
  int compressor_id() { return compressor_id_; }

 private:

  // A callback helper for AddToArchive.
  void AddToArchiveCallback(int32_t, const pp::VarDictionary& dictionary);

  // A callback helper for CloseArchive.
  void CloseArchiveCallback(int32_t, bool has_error);

  // The compressor id of this compressor.
  int compressor_id_;

  // An object that sends messages to JavaScript.
  JavaScriptMessageSenderInterface* message_sender_;

  // A worker for jobs that require blocking operations or a lot of processing
  // time. Those shouldn't be done on the main thread. The jobs submitted to
  // this thread are executed in order, so a new job must wait for the last job
  // to finish.
  pp::SimpleThread worker_;

  // Callback factory used to submit jobs to worker_.
  pp::CompletionCallbackFactory<Compressor> callback_factory_;

  // A requestor for making calls to JavaScript.
  JavaScriptCompressorRequestorInterface* requestor_;

  // Libarchive wrapper instance per compressor, shared across all operations.
  CompressorArchive* compressor_archive_;

  // An instance that takes care of all IO operations.
  CompressorStream* compressor_stream_;
};

#endif  /// COMPRESSOR_H_
