# @HEADER
# ************************************************************************
#
#            TriBITS: Tribial Build, Integrate, and Test System
#                    Copyright 2013 Sandia Corporation
#
# Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
# the U.S. Government retains certain rights in this software.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. Neither the name of the Corporation nor the names of the
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# ************************************************************************
# @HEADER


INCLUDE(TribitsProcessPackagesAndDirsLists)

INCLUDE(TribitsAddOptionAndDefine)
INCLUDE(AdvancedOption)
INCLUDE(AdvancedSet)
INCLUDE(AppendStringVar)
INCLUDE(CMakeBuildTypesList)
INCLUDE(FindListElement)
INCLUDE(GlobalNullSet)
INCLUDE(PrintNonemptyVar)
INCLUDE(PrintVar)
INCLUDE(RemoveGlobalDuplicates)
INCLUDE(SetDefault)
INCLUDE(MessageWrapper)
INCLUDE(DualScopeSet)

#
# Function that creates error message about missing/misspelled package.
#

FUNCTION(TRIBITS_ABORT_ON_MISSING_PACKAGE   DEP_PKG  PACKAGE_NAME  DEP_PKG_LIST_NAME)
  MULTILINE_SET(ERRMSG
    "Error, the package '${DEP_PKG}' is listed as a dependency of the package"
    " '${PACKAGE_NAME}' is in the list '${DEP_PKG_LIST_NAME}' but the package"
    " '${DEP_PKG}' is either not defined or is listed later in the package order."
    "  Check the spelling of '${DEP_PKG}' or see how it is listed in"
    " ${PROJECT_NAME}_PACKAGES_AND_DIRS_AND_CLASSIFICATIONS in relationship to"
    " '${PACKAGE_NAME}'.")
  MESSAGE(FATAL_ERROR ${ERRMSG})
ENDFUNCTION()


FUNCTION(TRIBITS_ABORT_ON_SELF_DEP  PACKAGE_NAME  DEP_PKG_LIST_NAME)
  MULTILINE_SET(ERRMSG
    "Error, the package '${PACKAGE_NAME}' is listed as a dependency itself"
    " in the list '${DEP_PKG_LIST_NAME}'!")
  MESSAGE(FATAL_ERROR ${ERRMSG})
ENDFUNCTION()


#
# Function that helps to set up backward package dependency lists
#

FUNCTION(TRIBITS_SET_DEP_PACKAGES  PACKAGE_NAME   LIB_OR_TEST  REQUIRED_OR_OPTIONAL)

  #MESSAGE("\nPACKAGE_ARCH_SET_DEP_PACKAGES: ${PACKAGE_NAME}"
  #  "  ${LIB_OR_TEST}  ${REQUIRED_OR_OPTIONAL})")

  SET(LIST_TYPE  ${LIB_OR_TEST}_${REQUIRED_OR_OPTIONAL}_DEP_PACKAGES)

  SET(PACKAGE_DEPS_LIST)

  FOREACH(DEP_PKG ${${LIST_TYPE}})
    IF (${DEP_PKG} STREQUAL ${PACKAGE_NAME})
      TRIBITS_ABORT_ON_SELF_DEP("${PACKAGE_NAME}" "${LIST_TYPE}")
    ENDIF()
    FIND_LIST_ELEMENT( ${PROJECT_NAME}_SE_PACKAGES ${DEP_PKG} DEP_PKG_FOUND)
    #PRINT_VAR(DEP_PKG_FOUND)
    IF (DEP_PKG_FOUND)
      LIST(APPEND PACKAGE_DEPS_LIST ${DEP_PKG})
    ELSE()
      IF (${PROJECT_NAME}_ASSERT_MISSING_PACKAGES AND NOT ${DEP_PKG}_ALLOW_MISSING_EXTERNAL_PACKAGE)
        TRIBITS_ABORT_ON_MISSING_PACKAGE("${DEP_PKG}" "${PACKAGE_NAME}" "${PROJECT_NAME}_SE_PACKAGES")
      ELSE()
        IF (${DEP_PKG}_ALLOW_MISSING_EXTERNAL_PACKAGE)
          MESSAGE_WRAPPER("WARNING: ${DEP_PKG} is being ignored since its directory"
            " is missing and ${DEP_PKG}_ALLOW_MISSING_EXTERNAL_PACKAGE ="
            " ${${DEP_PKG}_ALLOW_MISSING_EXTERNAL_PACKAGE}!")
          IF (REQUIRED_OR_OPTIONAL STREQUAL "REQUIRED")
            SET(SE_PACKAGE_ENABLE_VAR  ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
            MESSAGE_WRAPPER("WARNING: Setting ${SE_PACKAGE_ENABLE_VAR}=OFF because"
              " ${DEP_PKG} is a required missing package!") 
            DUAL_SCOPE_SET(${SE_PACKAGE_ENABLE_VAR} OFF)
          ENDIF()
        ENDIF()
        IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
          MESSAGE(
            "\n***"
            "\n*** WARNING: The package ${DEP_PKG} which is a dependent package of"
              " ${PACKAGE_NAME} being ignored because ${DEP_PKG} is missing!"
            "\n***\n" )
        ENDIF()
      ENDIF()
    ENDIF()
  ENDFOREACH()

  #PRINT_VAR(PACKAGE_DEPS_LIST)

  GLOBAL_SET(${PACKAGE_NAME}_${LIST_TYPE} ${PACKAGE_DEPS_LIST})

ENDFUNCTION()


#
# Function that sets up the full package dependencies for each package
# This is needed by the export system to determine what libraries are 
# needed and for informational purposes for dependent projects.
#
FUNCTION(TRIBITS_SET_FULL_DEP_PACKAGES PACKAGE_NAME)

  IF(${PROJECT_NAME}_GENERATE_EXPORT_FILE_DEPENDENCIES)
    #message("{")
    LIST(FIND ${PROJECT_NAME}_SE_PACKAGES ${PACKAGE_NAME} PACKAGE_VALUE)
    #message("Entering TRIBITS_SET_FULL_DEP_PACKAGES for package '${PACKAGE_NAME}'(${PACKAGE_VALUE})")

    LIST(APPEND PACKAGE_FULL_DEPS_LIST ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES}
      ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES} )

    IF(PACKAGE_FULL_DEPS_LIST)
      LIST(REMOVE_DUPLICATES PACKAGE_FULL_DEPS_LIST)

      FOREACH(DEP_PACKAGE ${PACKAGE_FULL_DEPS_LIST})
        LIST(APPEND PACKAGE_FULL_DEPS_LIST ${${DEP_PACKAGE}_FULL_EXPORT_DEP_PACKAGES})
      ENDFOREACH()

      LIST(REMOVE_DUPLICATES PACKAGE_FULL_DEPS_LIST)
    ENDIF()

    SET(ORDERED_PACKAGE_FULL_DEPS_LIST)

    FOREACH(DEP_PACKAGE ${PACKAGE_FULL_DEPS_LIST})
      LIST(FIND ${PROJECT_NAME}_SE_PACKAGES ${DEP_PACKAGE} DEP_PACKAGE_VALUE)
      #message("inserting package ${DEP_PACKAGE}(${DEP_PACKAGE_VALUE}) into full dep list for ${PACKAGE_NAME}")
      SET(SORTED_INDEX 0)
      SET(INSERTED_DEP_PACKAGE FALSE)
      FOREACH(SORTED_PACKAGE ${ORDERED_PACKAGE_FULL_DEPS_LIST})
        LIST(FIND ${PROJECT_NAME}_SE_PACKAGES ${SORTED_PACKAGE} SORTED_PACKAGE_VALUE)
        IF(${DEP_PACKAGE_VALUE} GREATER ${SORTED_PACKAGE_VALUE})
          #message("inserted package ${DEP_PACKAGE} at index ${SORTED_INDEX}")
          LIST(INSERT ORDERED_PACKAGE_FULL_DEPS_LIST ${SORTED_INDEX} ${DEP_PACKAGE})
          SET(INSERTED_DEP_PACKAGE TRUE)
          BREAK()
        ENDIF()
        MATH(EXPR SORTED_INDEX ${SORTED_INDEX}+1)
      ENDFOREACH()
      IF(NOT INSERTED_DEP_PACKAGE)
        #message("inserted package ${DEP_PACKAGE} at the end of the list (index = ${SORTED_INDEX})")
        LIST(APPEND ORDERED_PACKAGE_FULL_DEPS_LIST ${DEP_PACKAGE})
      ENDIF()    
    ENDFOREACH()

    #message("Full lib package dependencies for package        ${PACKAGE_NAME} = '${PACKAGE_FULL_DEPS_LIST}'")
    #message("Full sorted lib package dependencies for package ${PACKAGE_NAME} = '${ORDERED_PACKAGE_FULL_DEPS_LIST}'")
    GLOBAL_SET(${PACKAGE_NAME}_FULL_EXPORT_DEP_PACKAGES ${ORDERED_PACKAGE_FULL_DEPS_LIST})

    #message("Leaving TRIBITS_SET_FULL_DEP_PACKAGES for package '${PACKAGE_NAME}'")
    #message("}\n")
  ENDIF()
ENDFUNCTION()

#
# Macro that helps to set up forward package dependency lists
#

FUNCTION(TRIBITS_APPEND_FORWARD_DEP_PACKAGES PACKAGE_NAME LIST_TYPE)
 
  #MESSAGE("\nPACKAGE_ARCH_APPEND_FORWARD_DEP_PACKAGES: ${PACKAGE_NAME} ${LIST_TYPE}")

  SET(DEP_PKG_LIST_NAME "${PACKAGE_NAME}_${LIST_TYPE}")

  #MESSAGE("DEP_PKG_LIST_NAME = ${DEP_PKG_LIST_NAME}")
  #MESSAGE("${DEP_PKG_LIST_NAME} = ${${DEP_PKG_LIST_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ASSERT_MISSING_PACKAGES)
  FOREACH(DEP_PKG ${${DEP_PKG_LIST_NAME}})
    #MESSAGE("DEP_PKG = ${DEP_PKG}")
    SET(FWD_DEP_PKG_LIST_NAME "${DEP_PKG}_FORWARD_${LIST_TYPE}")
    #MESSAGE("FWD_DEP_PKG_LIST_NAME = ${FWD_DEP_PKG_LIST_NAME}")
    IF (NOT DEFINED ${FWD_DEP_PKG_LIST_NAME})
      IF (${PROJECT_NAME}_ASSERT_MISSING_PACKAGES)
        TRIBITS_ABORT_ON_MISSING_PACKAGE(${DEP_PKG} ${PACKAGE_NAME} ${DEP_PKG_LIST_NAME})
      ELSE()
        IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
          MESSAGE(
            "\n***"
            "\n*** WARNING: The package ${DEP_PKG} has forward dependent package"
              " ${PACKAGE_NAME}, but that dependency is being ignored because the package"
              " ${DEP_PKG} is missing!"
            "\n***\n" )
        ENDIF()
      ENDIF()
    ELSE()
      SET(${FWD_DEP_PKG_LIST_NAME} ${${FWD_DEP_PKG_LIST_NAME}} ${PACKAGE_NAME} PARENT_SCOPE)
    ENDIF()
  ENDFOREACH()

ENDFUNCTION()


################################################################################
#
# Helper macros for TRIBITS_READ_PACKAGE_DEPENDENCIES()


MACRO(TRIBITS_PREP_TO_READ_DEPENDENCIES)

  DECLARE_UNDEFINED(LIB_REQUIRED_DEP_PACKAGES)
  DECLARE_UNDEFINED(LIB_OPTIONAL_DEP_PACKAGES)
  DECLARE_UNDEFINED(TEST_REQUIRED_DEP_PACKAGES)
  DECLARE_UNDEFINED(TEST_OPTIONAL_DEP_PACKAGES)

  DECLARE_UNDEFINED(LIB_REQUIRED_DEP_TPLS "")
  DECLARE_UNDEFINED(LIB_OPTIONAL_DEP_TPLS "")
  DECLARE_UNDEFINED(TEST_REQUIRED_DEP_TPLS "")
  DECLARE_UNDEFINED(TEST_OPTIONAL_DEP_TPLS "")

ENDMACRO()


MACRO(TRIBITS_SAVE_OFF_DEPENENCIES_VARS  POSTFIX)

  SET(LIB_REQUIRED_DEP_PACKAGES_${POSTFIX} ${LIB_REQUIRED_DEP_PACKAGES})
  SET(LIB_OPTIONAL_DEP_PACKAGES_${POSTFIX} ${LIB_OPTIONAL_DEP_PACKAGES})
  SET(TEST_REQUIRED_DEP_PACKAGES_${POSTFIX} ${TEST_REQUIRED_DEP_PACKAGES})
  SET(TEST_OPTIONAL_DEP_PACKAGES_${POSTFIX} ${TEST_OPTIONAL_DEP_PACKAGES})

  SET(LIB_REQUIRED_DEP_TPLS_${POSTFIX} ${LIB_REQUIRED_DEP_TPLS})
  SET(LIB_OPTIONAL_DEP_TPLS_${POSTFIX} ${LIB_OPTIONAL_DEP_TPLS})
  SET(TEST_REQUIRED_DEP_TPLS_${POSTFIX} ${TEST_REQUIRED_DEP_TPLS})
  SET(TEST_OPTIONAL_DEP_TPLS_${POSTFIX} ${TEST_OPTIONAL_DEP_TPLS})

ENDMACRO()


MACRO(TRIBITS_ASSERT_READ_DEPENDENCY_VARS  PACKAGE_NAME)

  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(LIB_REQUIRED_DEP_PACKAGES ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(LIB_OPTIONAL_DEP_PACKAGES ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(TEST_REQUIRED_DEP_PACKAGES ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(TEST_OPTIONAL_DEP_PACKAGES ${PACKAGE_NAME})

  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(LIB_REQUIRED_DEP_TPLS ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(LIB_OPTIONAL_DEP_TPLS ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(TEST_REQUIRED_DEP_TPLS ${PACKAGE_NAME})
  TRIBITS_ASSERT_DEFINED_PACKAGE_VAR(TEST_OPTIONAL_DEP_TPLS ${PACKAGE_NAME})

ENDMACRO()


MACRO(TRIBITS_READ_BACK_DEPENDENCIES_VARS  POSTFIX)

  SET(LIB_REQUIRED_DEP_PACKAGES ${LIB_REQUIRED_DEP_PACKAGES_${POSTFIX}})
  SET(LIB_OPTIONAL_DEP_PACKAGES ${LIB_OPTIONAL_DEP_PACKAGES_${POSTFIX}})
  SET(TEST_REQUIRED_DEP_PACKAGES ${TEST_REQUIRED_DEP_PACKAGES_${POSTFIX}})
  SET(TEST_OPTIONAL_DEP_PACKAGES ${TEST_OPTIONAL_DEP_PACKAGES_${POSTFIX}})

  SET(LIB_REQUIRED_DEP_TPLS ${LIB_REQUIRED_DEP_TPLS_${POSTFIX}})
  SET(LIB_OPTIONAL_DEP_TPLS ${LIB_OPTIONAL_DEP_TPLS_${POSTFIX}})
  SET(TEST_REQUIRED_DEP_TPLS ${TEST_REQUIRED_DEP_TPLS_${POSTFIX}})
  SET(TEST_OPTIONAL_DEP_TPLS ${TEST_OPTIONAL_DEP_TPLS_${POSTFIX}})

ENDMACRO()


MACRO(TRIBITS_PROCESS_PACKAGE_DEPENDENCIES_LISTS  PACKAGE_NAME)

  TRIBITS_SET_DEP_PACKAGES(${PACKAGE_NAME} LIB  REQUIRED)
  TRIBITS_SET_DEP_PACKAGES(${PACKAGE_NAME} LIB  OPTIONAL)
  TRIBITS_SET_DEP_PACKAGES(${PACKAGE_NAME} TEST  REQUIRED)
  TRIBITS_SET_DEP_PACKAGES(${PACKAGE_NAME} TEST  OPTIONAL)
  
  TRIBITS_SET_FULL_DEP_PACKAGES(${PACKAGE_NAME})

  SET(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS ${LIB_REQUIRED_DEP_TPLS})
  SET(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS ${LIB_OPTIONAL_DEP_TPLS})
  SET(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS ${TEST_REQUIRED_DEP_TPLS})
  SET(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS ${TEST_OPTIONAL_DEP_TPLS})

  TRIBITS_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} LIB_REQUIRED_DEP_PACKAGES)
  TRIBITS_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} LIB_OPTIONAL_DEP_PACKAGES)
  TRIBITS_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} TEST_REQUIRED_DEP_PACKAGES)
  TRIBITS_APPEND_FORWARD_DEP_PACKAGES(${PACKAGE_NAME} TEST_OPTIONAL_DEP_PACKAGES)

ENDMACRO()


#
# Parse the read-in varaible SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS, add
# read subpackages to list of defined SE packages, and define options.
#
# NOTE: Directly reads varaibles ${PACKAGE_NAME} and
# ${SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS} defined in
# TRIBITS_READ_PACKAGE_DEPENDENCIES
#
MACRO(TRIBITS_PARSE_SUBPACKAGES_AND_APPEND_SE_PACKAGES_AND_ADD_OPTIONS  PACKAGE_NAME)

  #MESSAGE("TRIBITS_PARSE_SUBPACKAGES_AND_APPEND_SE_PACKAGES_AND_ADD_OPTIONS: ${PACKAGE_NAME}")

  # Structure of SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS
  SET(SPDC_SP_NAME_OFFSET 0)
  SET(SPDC_SP_DIR_OFFSET 1)
  SET(SPDC_SP_CLASSIFICATION_OFFSET 2)
  SET(SPDC_SP_OPTREQ_OFFSET 3)
  SET(SPDC_NUM_FIELDS 4)

  SET(${PACKAGE_NAME}_SUBPACKAGES)
  SET(${PACKAGE_NAME}_SUBPACKAGE_DIRS)
  SET(${PACKAGE_NAME}_SUBPACKAGE_OPTREQ)

  IF (SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS)

    LIST(LENGTH SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS SPDC_TOTAL_LENGTH)
    MATH(EXPR NUM_SUBPACKAGES "${SPDC_TOTAL_LENGTH}/${SPDC_NUM_FIELDS}")
    #PRINT_VAR(NUM_SUBPACKAGES)
    MATH(EXPR SUBPACKAGES_LAST_IDX "${NUM_SUBPACKAGES}-1")
    #PRINT_VAR(SUBPACKAGES_LAST_IDX)
  
    FOREACH(SUBPACKAGE_IDX RANGE ${SUBPACKAGES_LAST_IDX})
  
      #MESSAGE("")
      #PRINT_VAR(SUBPACKAGE_IDX)

      # SUBPACKAGE_NAME
      MATH(EXPR SUBPACKAGE_NAME_IDX "${SUBPACKAGE_IDX}*${SPDC_NUM_FIELDS}+${SPDC_SP_NAME_OFFSET}")
      #PRINT_VAR(SUBPACKAGE_NAME_IDX)
      LIST(GET SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS ${SUBPACKAGE_NAME_IDX} SUBPACKAGE_NAME )
      #PRINT_VAR(SUBPACKAGE_NAME)
 
      # SUBPACKAGE_DIR     
      MATH(EXPR SUBPACKAGE_DIR_IDX "${SUBPACKAGE_IDX}*${SPDC_NUM_FIELDS}+${SPDC_SP_DIR_OFFSET}")
      #PRINT_VAR(SUBPACKAGE_DIR_IDX)
      LIST(GET SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS ${SUBPACKAGE_DIR_IDX} SUBPACKAGE_DIR )
      #PRINT_VAR(SUBPACKAGE_DIR)
 
      # SUBPACKAGE_CLASSIFICATION     
      MATH(EXPR SUBPACKAGE_CLASSIFICATION_IDX
        "${SUBPACKAGE_IDX}*${SPDC_NUM_FIELDS}+${SPDC_SP_CLASSIFICATION_OFFSET}")
      #PRINT_VAR(SUBPACKAGE_CLASSIFICATION_IDX)
      LIST(GET SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS ${SUBPACKAGE_CLASSIFICATION_IDX}
        SUBPACKAGE_CLASSIFICATION )
      #PRINT_VAR(SUBPACKAGE_CLASSIFICATION)
 
      # SUBPACKAGE_OPTREQ     
      MATH(EXPR SUBPACKAGE_OPTREQ_IDX
        "${SUBPACKAGE_IDX}*${SPDC_NUM_FIELDS}+${SPDC_SP_OPTREQ_OFFSET}")
      #PRINT_VAR(SUBPACKAGE_OPTREQ_IDX)
      LIST(GET SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS ${SUBPACKAGE_OPTREQ_IDX}
        SUBPACKAGE_OPTREQ )
      #PRINT_VAR(SUBPACKAGE_OPTREQ)

      SET(SUBPACKAGE_FULLNAME ${PACKAGE_NAME}${SUBPACKAGE_NAME})

      # Append to lists and global varibles

      LIST(APPEND ${PACKAGE_NAME}_SUBPACKAGES ${SUBPACKAGE_NAME})
      LIST(APPEND ${PACKAGE_NAME}_SUBPACKAGE_DIRS ${SUBPACKAGE_DIR})
      LIST(APPEND ${PACKAGE_NAME}_SUBPACKAGE_OPTREQ ${SUBPACKAGE_OPTREQ})
      LIST(APPEND ${PROJECT_NAME}_SE_PACKAGES ${SUBPACKAGE_FULLNAME})
      SET(${SUBPACKAGE_FULLNAME}_PARENT_PACKAGE ${PACKAGE_NAME})

      # Set up the input options for this subpackage
      TRIBITS_INSERT_STANDARD_PACKAGE_OPTIONS(${SUBPACKAGE_FULLNAME}
        ${SUBPACKAGE_CLASSIFICATION})

      #PRINT_VAR(${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME})
  
    ENDFOREACH()

  ENDIF()

  #PRINT_VAR(${PACKAGE_NAME}_SUBPACKAGES)
  #PRINT_VAR(${PACKAGE_NAME}_SUBPACKAGE_OPTREQ)

ENDMACRO()


#
# Macro that reads in a single subpackage dependencies file and sets up
# the dependency structure for it.
#
MACRO(TRIBITS_READ_SUBPACKAGE_DEPENDENCIES  PACKAGE_NAME  PACKAGE_DIR
  SUBPACKAGE_NAME  SUBPACKAGE_DIR)

  #MESSAGE("TRIBITS_READ_SUBPACKAGE_DEPENDENCIES: ${PACKAGE_NAME} ${PACKAGE_DIR} ${SUBPACKAGE_NAME} ${SUBPACKAGE_DIR}")

  SET(SUBPACKAGE_FULLNAME ${PACKAGE_NAME}${SUBPACKAGE_NAME})

  #
  # A) Get ready to read in the contents of this this subpakages's Dependencies.cmake file
  #

  SET(${SUBPACKAGE_FULLNAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES "")
  SET(${SUBPACKAGE_FULLNAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES "")
  SET(${SUBPACKAGE_FULLNAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES "")
  SET(${SUBPACKAGE_FULLNAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES "")

  TRIBITS_PREP_TO_READ_DEPENDENCIES()

  # NOTE: Subpackages use the regression email list from the parent package.

  # NOTE: Subpackages are not allowed to have subpackages!
  SET(SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS)

  #
  # B) Read in this subpackage's Dependencies file
  #

  SET(SUBPACKAGE_FULL_DIR "${PACKAGE_DIR}/${SUBPACKAGE_DIR}")
  LIST(APPEND ${PROJECT_NAME}_SE_PACKAGE_DIRS ${SUBPACKAGE_FULL_DIR}) 

  SET(SUBPACKAGE_ABS_DIR "${PROJECT_SOURCE_DIR}/${SUBPACKAGE_FULL_DIR}")
  SET(SUBPAKCAGE_DEPENDENCIES_FILE "${SUBPACKAGE_ABS_DIR}/cmake/Dependencies.cmake")

  INCLUDE(${SUBPAKCAGE_DEPENDENCIES_FILE})

  TRIBITS_ASSERT_READ_DEPENDENCY_VARS(${SUBPACKAGE_FULLNAME})

  #
  # C) Finish processing this subpackage's dependencies into dependency graph vars
  #

  TRIBITS_PROCESS_PACKAGE_DEPENDENCIES_LISTS(${SUBPACKAGE_FULLNAME})

  #PRINT_VAR(${SUBPACKAGE_FULLNAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES)
  #PRINT_VAR(${SUBPACKAGE_FULLNAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES)
  #PRINT_VAR(${SUBPACKAGE_FULLNAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES)
  #PRINT_VAR(${SUBPACKAGE_FULLNAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES)

  #
  # D) Set the email addresses for the subpackage to the parent package's
  #

  SET(${SUBPACKAGE_FULLNAME}_REGRESSION_EMAIL_LIST ${${PACKAGE_NAME}_REGRESSION_EMAIL_LIST})

ENDMACRO()


#
# Read in subpackages dependencies files and add to dependencies graph varibles
#
MACRO(TRIBITS_READ_ALL_PACKAGE_SUBPACKAGE_DEPENDENCIES  PACKAGE_NAME  PACKAGE_DIR)

  #MESSAGE("TRIBITS_READ_ALL_PACKAGE_SUBPACKAGE_DEPENDENCIES: ${PACAKGE_NAME} ${PACKAGE_DIR}")

  #PRINT_VAR(${PROJECT_NAME}_SE_PACKAGES)

  SET(SUBPACKAGE_IDX 0)
  FOREACH(TRIBITS_SUBPACKAGE ${${PACKAGE_NAME}_SUBPACKAGES})
    LIST(GET ${PACKAGE_NAME}_SUBPACKAGE_DIRS ${SUBPACKAGE_IDX} SUBPACKAGE_DIR)
    TRIBITS_READ_SUBPACKAGE_DEPENDENCIES(${TRIBITS_PACKAGE}  ${PACKAGE_DIR}
      ${TRIBITS_SUBPACKAGE}  ${SUBPACKAGE_DIR})
    MATH(EXPR SUBPACKAGE_IDX "${SUBPACKAGE_IDX}+1")
  ENDFOREACH()

  LIST(APPEND ${PROJECT_NAME}_SE_PACKAGE_DIRS ${PACKAGE_DIR}) 

ENDMACRO()


#
# Macro that reads in package dependencies for a package and sets forward
# dependencies for packages already read in.
#
# Modifies the global variables:
#
#   ${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES
#   ${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES
#
MACRO(TRIBITS_READ_PACKAGE_DEPENDENCIES  PACKAGE_NAME  PACKAGE_DIR)

  #
  # A) Get ready to read in the contents of this this pakages's Dependencies.cmake file
  #

  SET(${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES "")
  SET(${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES "")

  TRIBITS_PREP_TO_READ_DEPENDENCIES()

  # Set one regression email list for the package and all subpackages!
  SET(REGRESSION_EMAIL_LIST "") # Allow to be empty

  # Listing of subpakages
  SET(SUBPACKAGES_DIRS_CLASSIFICATIONS_OPTREQS) # Allow to be empty

  #
  # B) Read in this package's Dependencies file and save off read dependency vars.
  #

  SET(PAKCAGE_DEPENDENCIES_FILE
    "${PROJECT_SOURCE_DIR}/${PACKAGE_DIR}/cmake/Dependencies.cmake")

  INCLUDE(${PAKCAGE_DEPENDENCIES_FILE})

  TRIBITS_ASSERT_READ_DEPENDENCY_VARS(${PACKAGE_NAME})

  TRIBITS_SAVE_OFF_DEPENENCIES_VARS(PARENTPACK)

  #
  # B.1) Set up the mail addresses
  #

  # ToDo: Move this above so that it will be handled as part of subpackage
  # processing?

  # Lower-case package name To be used with auto email naming based on base email address
  STRING(TOLOWER "${PACKAGE_NAME}" LPACKAGE)
  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    PRINT_VAR(REGRESSION_EMAIL_LIST)
  ENDIF()
 
  SET(REPOSITORY_NAME ${${PACKAGE_NAME}_PARENT_REPOSITORY})

  IF (REGRESSION_EMAIL_LIST AND NOT ${REPOSITORY_NAME}_REPOSITORY_OVERRIDE_PACKAGE_EMAIL_LIST)
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST ${REGRESSION_EMAIL_LIST})
  ELSEIF (${REPOSITORY_NAME}_REPOSITORY_EMAIL_URL_ADDRESSS_BASE)
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST
      "${LPACKAGE}-regression@${${REPOSITORY_NAME}_REPOSITORY_EMAIL_URL_ADDRESSS_BASE}")
  ELSEIF (${REPOSITORY_NAME}_REPOSITORY_MASTER_EMAIL_ADDRESSS)
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST
      "${${REPOSITORY_NAME}_REPOSITORY_MASTER_EMAIL_ADDRESSS}")
  ELSEIF (${PROJECT_NAME}_PROJECT_EMAIL_URL_ADDRESSS_BASE)
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST
      "${LPACKAGE}-regression@${${PROJECT_NAME}_PROJECT_EMAIL_URL_ADDRESSS_BASE}")
  ELSEIF (${PROJECT_NAME}_PROJECT_MASTER_EMAIL_ADDRESSS)
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST
      "${${PROJECT_NAME}_PROJECT_MASTER_EMAIL_ADDRESSS}")
  ELSE()
    SET(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST "")
  ENDIF()

  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    PRINT_VAR(${PACKAGE_NAME}_REGRESSION_EMAIL_LIST)
  ENDIF()

  #
  # B.2) Process this package's subpackages first *before* finishing this packages!
  #

  TRIBITS_PARSE_SUBPACKAGES_AND_APPEND_SE_PACKAGES_AND_ADD_OPTIONS(${PACKAGE_NAME})

  TRIBITS_READ_ALL_PACKAGE_SUBPACKAGE_DEPENDENCIES(${PACKAGE_NAME} ${PACKAGE_DIR})

  #
  # C) Finish processing this package's dependencies into dependency graph vars
  #
  # NOTE: The subpackages for this package are automatically treated as
  # optional or required library dependent packages for this outer package!
  #

  TRIBITS_READ_BACK_DEPENDENCIES_VARS(PARENTPACK)

  # Append the subpackages to the dependencies list
  SET(SUBPACKAGE_IDX 0)
  FOREACH(TRIBITS_SUBPACKAGE ${${PACKAGE_NAME}_SUBPACKAGES})
    SET(SUBPACKAGE_FULLNAME ${PACKAGE_NAME}${TRIBITS_SUBPACKAGE})
    LIST(GET ${PACKAGE_NAME}_SUBPACKAGE_OPTREQ ${SUBPACKAGE_IDX} SUBPACKAGE_OPTREQ)
    LIST(APPEND LIB_${SUBPACKAGE_OPTREQ}_DEP_PACKAGES ${SUBPACKAGE_FULLNAME})
    MATH(EXPR SUBPACKAGE_IDX "${SUBPACKAGE_IDX}+1")
  ENDFOREACH()

  # Append this package to list of SE pacakges *after* subpackages are added!
  LIST(APPEND ${PROJECT_NAME}_SE_PACKAGES ${PACKAGE_NAME})

  # Process this parent package's dependency lists!
  TRIBITS_PROCESS_PACKAGE_DEPENDENCIES_LISTS(${PACKAGE_NAME})

ENDMACRO()


#
# Get the REPO_NAME and REPO_DIR given the REPO
#

FUNCTION(TRIBITS_GET_REPO_NAME_DIR  REPO_IN  REPO_NAME_OUT  REPO_DIR_OUT)
  #MESSAGE("TRIBITS_GET_REPO_NAME_DIR:  '${REPO_IN}'  '${REPO_NAME_OUT}'  '${REPO_DIR_OUT}'")
  # This list of repositories is the list of directories!
  SET(REPO_DIR ${REPO_IN})
  # Get the Repository name
  IF (REPO_IN STREQUAL ".")
    # The Project and the Reposiotry are one and the same
    SET(REPO_NAME ${PROJECT_NAME})
  ELSE()
    # The Repository name is the same as the repository directory
    SET(REPO_NAME ${REPO_IN})
  ENDIF()
  SET(${REPO_NAME_OUT} ${REPO_NAME} PARENT_SCOPE)
  SET(${REPO_DIR_OUT} ${REPO_DIR} PARENT_SCOPE)
ENDFUNCTION()


#
# Macro that reads all the package dependencies and builds dependency graph
#
# Reads from the variables:
#   ${PROJECT_NAME}_ALL_REPOSITORIES
#   ${PROJECT_NAME}_PACKAGES
#
# Writes to:
#   ${PROJECT_NAME}_SE_PACKAGES
#   ${PROJECT_NAME}_SE_PACKAGES_DIRS	
#

MACRO(TRIBITS_READ_ALL_PACKAGE_DEPENDENCIES)

  MESSAGE("")
  MESSAGE("Processing Project, Repository, and Package dependency files and building internal dependencies graph ...")
  MESSAGE("")

  #
  # A) First, process the Repository and Project dependency files
  #

  FOREACH(TIBITS_REPO ${${PROJECT_NAME}_ALL_REPOSITORIES})
    TRIBITS_GET_REPO_NAME_DIR(${TIBITS_REPO}  REPO_NAME  REPO_DIR)
    #PRINT_VAR(REPO_NAME)
    #PRINT_VAR(REPO_DIR)
    SET(REPO_DEPENDENCIES_SETUP_FILE
      "${PROJECT_SOURCE_DIR}/${REPO_DIR}/cmake/RepositoryDependenciesSetup.cmake")
    #PRINT_VAR(REPO_DEPENDENCIES_SETUP_FILE)
    IF (EXISTS ${REPO_DEPENDENCIES_SETUP_FILE})
      IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
        MESSAGE("-- " "Processing ${REPO_NAME} file ${REPO_DEPENDENCIES_SETUP_FILE} ...")
      ENDIF()
      INCLUDE(${REPO_DEPENDENCIES_SETUP_FILE})
      IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
        PRINT_VAR(${REPO_NAME}_REPOSITORY_EMAIL_URL_ADDRESSS_BASE)
        PRINT_VAR(${REPO_NAME}_REPOSITORY_MASTER_EMAIL_ADDRESSS)
      ENDIF()
    ELSE()
      IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
        MESSAGE("-- " "The ${REPO_NAME} file ${REPO_DEPENDENCIES_SETUP_FILE} does not exist! ...")
      ENDIF()
    ENDIF()
  ENDFOREACH()

  SET(PROJECT_DEPENDENCIES_SETUP_FILE
    "${PROJECT_SOURCE_DIR}/cmake/ProjectDependenciesSetup.cmake")
  IF (EXISTS ${PROJECT_DEPENDENCIES_SETUP_FILE})
    IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
      MESSAGE("-- " "Processing ${PROJECT_NAME} file ${PROJECT_DEPENDENCIES_SETUP_FILE} ...")
    ENDIF()
    INCLUDE(${PROJECT_DEPENDENCIES_SETUP_FILE})
    IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
      PRINT_VAR(${PROJECT_NAME}_PROJECT_EMAIL_URL_ADDRESSS_BASE)
      PRINT_VAR(${PROJECT_NAME}_PROJECT_MASTER_EMAIL_ADDRESSS)
    ENDIF()
  ELSE()
    IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
      MESSAGE("-- " "The ${PROJECT_NAME} file ${PROJECT_DEPENDENCIES_SETUP_FILE} does not exist! ...")
    ENDIF()
  ENDIF()

  #
  # B) Process the package dependency files, yielding the list of subpackages as well
  #
  
  SET(${PROJECT_NAME}_SE_PACKAGES) # Packages and subpackages
  SET(${PROJECT_NAME}_SE_PACKAGE_DIRS)
  
  SET(PACKAGE_IDX 0)
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_PACKAGES})
    LIST(GET ${PROJECT_NAME}_PACKAGE_DIRS ${PACKAGE_IDX} PACKAGE_DIR)
    TRIBITS_READ_PACKAGE_DEPENDENCIES(${TRIBITS_PACKAGE} ${PACKAGE_DIR})
    #TRIBITS_ADD_OPTIONAL_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
    MATH(EXPR PACKAGE_IDX "${PACKAGE_IDX}+1")
  ENDFOREACH()
  
  # Create a reverse se packages list for later use
  SET(${PROJECT_NAME}_REVERSE_SE_PACKAGES ${${PROJECT_NAME}_SE_PACKAGES})
  IF (${PROJECT_NAME}_REVERSE_SE_PACKAGES)
    LIST(REVERSE ${PROJECT_NAME}_REVERSE_SE_PACKAGES)
  ENDIF()

  LIST(LENGTH ${PROJECT_NAME}_SE_PACKAGES ${PROJECT_NAME}_NUM_SE_PACKAGES)
  PRINT_VAR(${PROJECT_NAME}_NUM_SE_PACKAGES)
  #PRINT_VAR(${PROJECT_NAME}_SE_PACKAGES)

  FOREACH(TPL ${${PROJECT_NAME}_TPLS})
    IF (TPL_TENTATIVE_ENABLE_${TPL})
      MESSAGE("-- Tentatively enabling TPL '${TPL}'")
      #PRINT_VAR(TPL_ENABLE_${TPL})
    ENDIF()
  ENDFOREACH()
 
  ADVANCED_OPTION(${PROJECT_NAME}_DUMP_PACKAGE_DEPENDENCIES
    "Dump the package dependency information."
    "${${PROJECT_NAME}_VERBOSE_CONFIGURE}" )

  IF (${PROJECT_NAME}_DUMP_PACKAGE_DEPENDENCIES)
    MESSAGE("")
    MESSAGE("Printing package dependencies ...")
    MESSAGE("")
    PRINT_VAR(${PROJECT_NAME}_PACKAGES)
    PRINT_VAR(${PROJECT_NAME}_SE_PACKAGES)
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
      TRIBITS_PRINT_PACKAGE_DEPENDENCIES(${TRIBITS_PACKAGE})
      MESSAGE("")
    ENDFOREACH()
  ENDIF()

ENDMACRO()


#
# Function that sets a varaible to DECLARED-UNDEFINED
#

FUNCTION(DECLARE_UNDEFINED VAR)
  SET(${VAR} DECLARED-UNDEFINED PARENT_SCOPE)
ENDFUNCTION()


#
# Function that asserts that a package dependency variable is defined
# correctly
#

FUNCTION(TRIBITS_ASSERT_DEFINED_PACKAGE_VAR PACKAGE_VAR PACKAGE_NAME)
  IF (${PACKAGE_VAR} STREQUAL DECLARED-UNDEFINED)
    MESSAGE(FATAL_ERROR
      "Error, the package variable ${PACKAGE_VAR} was not defined correctly for package ${PACKAGE_NAME}!"
      )
  ENDIF()
ENDFUNCTION()


#
# Private helper macros
#


FUNCTION(TRIBITS_PRIVATE_PRINT_DISABLE
  ENABLE_BEING_DISABLED_VAR_NAME  PACKAGE_WITH_SOMETHING_BEING_DISABLED
  DEP_TYPE_STR  THING_DISALBED_TYPE  THING_DISABLED_NAME
  )
  IF (${ENABLE_BEING_DISABLED_VAR_NAME})
    IF (${PROJECT_NAME}_DISABLE_ENABLED_FORWARD_DEP_PACKAGES)
      MESSAGE(
        " ***\n"
        " *** WARNING: Setting ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
        " which was 'ON' because ${PACKAGE_WITH_SOMETHING_BEING_DISABLED} has"
        " a required ${DEP_TYPE_STR} dependence on disabled"
        " ${THING_DISALBED_TYPE} ${THING_DISABLED_NAME}"
        " but ${PROJECT_NAME}_DISABLE_ENABLED_FORWARD_DEP_PACKAGES=ON!\n"
        " ***\n"
        )
    ELSE()
      MESSAGE(FATAL_ERROR
        " ***\n"
        " *** ERROR: Setting ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
        " which was 'ON' because ${PACKAGE_WITH_SOMETHING_BEING_DISABLED} has"
        " a required ${DEP_TYPE_STR} dependence on disabled"
        " ${THING_DISALBED_TYPE} ${THING_DISABLED_NAME}!\n"
        " ***\n"
        )
    ENDIF()
  ELSE()
    MESSAGE("-- "
      "Setting ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
      " because ${PACKAGE_WITH_SOMETHING_BEING_DISABLED} has a required ${DEP_TYPE_STR}"
      " dependence on disabled ${THING_DISALBED_TYPE} ${THING_DISABLED_NAME}")
  ENDIF()
ENDFUNCTION()


MACRO(TRIBITS_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE
  TPL_NAME PACKAGE_NAME LIBRARY_DEP
  )

  #MESSAGE("TRIBITS_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE"
  #  " ${TPL_NAME} ${PACKAGE_NAME} ${LIBRARY_DEP}")  

  # Only turn off PACKAGE_NAME libraries and test/eamples if it
  # is currently enabled or could be enabled.

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}
     OR ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} STREQUAL ""
     )

    IF ("${LIBRARY_DEP}" STREQUAL "TRUE")

      TRIBITS_PRIVATE_PRINT_DISABLE(
        ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} ${PACKAGE_NAME} "library"
        "TPL" ${TPL_NAME}
        )

      SET(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} OFF)

    ELSE()

      SET(DEP_TYPE_STR "test/example")

      ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_TESTS)
      IF (${PACKAGE_NAME}_ENABLE_TESTS
        OR ${PACKAGE_NAME}_ENABLE_TESTS STREQUAL ""
        )
        TRIBITS_PRIVATE_PRINT_DISABLE(
          ${PACKAGE_NAME}_ENABLE_TESTS ${PACKAGE_NAME} "${DEP_TYPE_STR}"
          "TPL" ${TPL_NAME}
          )
        SET(${PACKAGE_NAME}_ENABLE_TESTS OFF)
      ENDIF()

      ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_EXAMPLES)
      IF (${PACKAGE_NAME}_ENABLE_EXAMPLES
        OR ${PACKAGE_NAME}_ENABLE_EXAMPLES STREQUAL ""
        )
        TRIBITS_PRIVATE_PRINT_DISABLE(
          ${PACKAGE_NAME}_ENABLE_EXAMPLES ${PACKAGE_NAME} "${DEP_TYPE_STR}"
          "TPL" ${TPL_NAME}
          )
        SET(${PACKAGE_NAME}_ENABLE_EXAMPLES OFF)
      ENDIF()

    ENDIF()

  ENDIF()
  
ENDMACRO()


FUNCTION(TRIBITS_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE
  PACKAGE_NAME  PACKAGE_ENABLE_SOMETHING_VAR_NAME  FORWARD_DEP_PACKAGE_NAME
  DEP_TYPE_STR
  )
  TRIBITS_PRIVATE_PRINT_DISABLE(
    ${PACKAGE_ENABLE_SOMETHING_VAR_NAME} ${FORWARD_DEP_PACKAGE_NAME}
    "${DEP_TYPE_STR}" "package" ${PACKAGE_NAME} )
ENDFUNCTION()


MACRO(TRIBITS_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES
  FORWARD_DEP_PACKAGE_NAME PACKAGE_NAME LIBRARY_DEP
  )

  #MESSAGE("TRIBITS_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES"
  #  " ${FORWARD_DEP_PACKAGE_NAME} ${LIBRARY_DEP}")  

  # Only turn off FORWARD_DEP_PACKAGE libraries and test/eamples if it
  # is currently enabled or could be enabled

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}
     OR ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} STREQUAL ""
     )

    IF ("${LIBRARY_DEP}" STREQUAL "TRUE")

      TRIBITS_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}
        ${FORWARD_DEP_PACKAGE_NAME} "library" )

      SET(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} OFF)

    ELSE()

      SET(DEP_TYPE_STR "test/example")

      #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS)
      IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS
        OR ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS STREQUAL ""
        )
        TRIBITS_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
          ${PACKAGE_NAME} ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS
          ${FORWARD_DEP_PACKAGE_NAME} "${DEP_TYPE_STR}" )
        SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_TESTS OFF)
      ENDIF()
    
      #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES)
      IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES
        OR ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES STREQUAL ""
        )
        TRIBITS_PRIVATE_PRINT_DISABLE_REQUIRED_PACKAGE_ENABLE(
          ${PACKAGE_NAME} ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES
          ${FORWARD_DEP_PACKAGE_NAME} "${DEP_TYPE_STR}" )
        SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_EXAMPLES OFF)
      ENDIF()

    ENDIF()

  ENDIF()
  
ENDMACRO()


MACRO(TRIBITS_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES
  FORWARD_DEP_PACKAGE_NAME PACKAGE_NAME
  )

  #MESSAGE("TRIBITS_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES"
  #  " ${FORWARD_DEP_PACKAGE_NAME} ${PACKAGE_NAME}")
  #MESSAGE("-- " "${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} = ${${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}}")

  #ASSERT_DEFINED(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} OR "${${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}}" STREQUAL "")
    # Always disable the conditional enable but only print the message if the package is enabled.
    #MESSAGE("--  Disasble ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} ...")
    IF (${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
      MESSAGE("-- "
        "Setting ${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME}=OFF"
        " because ${FORWARD_DEP_PACKAGE_NAME} has an optional library dependence"
        " on disabled package ${PACKAGE_NAME}")
    ENDIF()
    SET(${FORWARD_DEP_PACKAGE_NAME}_ENABLE_${PACKAGE_NAME} OFF)
  ENDIF()

ENDMACRO()



#
# Macro that disables all packages and optional package TPL support
# for a disabled TPL.
#

MACRO(TRIBITS_DISABLE_TPL_DEP_PACKAGES TPL_NAME)

  #MESSAGE("TRIBITS_DISABLE_TPL_DEP_PACKAGES: ${TPL_NAME}")

  #PRINT_VAR(TPL_ENABLE_${TPL_NAME})
  IF ("${TPL_ENABLE_${TPL_NAME}}" STREQUAL "OFF")

    FOREACH(PACKAGE_NAME ${${PROJECT_NAME}_SE_PACKAGES})

      #ASSERT_DEFINED(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS)
      FOREACH(CURR_TPL_NAME ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS})
        IF (CURR_TPL_NAME STREQUAL ${TPL_NAME})
          TRIBITS_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE(
            ${TPL_NAME} ${PACKAGE_NAME} TRUE )
        ENDIF()
      ENDFOREACH()

      #ASSERT_DEFINED(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS)
      FOREACH(CURR_TPL_NAME ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS})
        IF (CURR_TPL_NAME STREQUAL ${TPL_NAME})
          TRIBITS_PRIVATE_DISABLE_TPL_REQUIRED_PACKAGE_ENABLE(
            ${TPL_NAME} ${PACKAGE_NAME} FALSE )
        ENDIF()
      ENDFOREACH()

    ENDFOREACH()

  ENDIF()

ENDMACRO()



#
# Macro that disables all of the subpackages of a parent package.
#

MACRO(TRIBITS_DISABLE_PARENTS_SUBPACKAGES PARENT_PACKAGE_NAME)

  #MESSAGE("TRIBITS_DISABLE_PARENTS_SUBPACKAGES: ${PARENT_PACKAGE_NAME}")

  #PRINT_VAR(${PROJECT_NAME}_ENABLE_${PARENT_PACKAGE_NAME})
  IF("${PROJECT_NAME}_ENABLE_${PARENT_PACKAGE_NAME}" STREQUAL "OFF")
  
    SET(SUBPACKAGE_IDX 0)
    FOREACH(TRIBITS_SUBPACKAGE ${${PARENT_PACKAGE_NAME}_SUBPACKAGES})
    
      SET(SUBPACKAGE_NAME ${TRIBITS_SUBPACKAGE})
      SET(SUBPACKAGE_FULLNAME ${PARENT_PACKAGE_NAME}${TRIBITS_SUBPACKAGE})
  
      #PRINT_VAR(${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME})
      IF (NOT "${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME}" STREQUAL "OFF")
        SET(ENABLE_BEING_DISABLED_VAR_NAME ${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME})
        MESSAGE("-- "
          "Setting subpackage enable ${ENABLE_BEING_DISABLED_VAR_NAME}=OFF"
          " because parent package ${PROJECT_NAME}_ENABLE_${PARENT_PACKAGE_NAME}=OFF")
        SET(${ENABLE_BEING_DISABLED_VAR_NAME} OFF)
      ENDIF()
    
      MATH(EXPR SUBPACKAGE_IDX "${SUBPACKAGE_IDX}+1")
    
    ENDFOREACH()

  ENDIF()

ENDMACRO()



#
# Function that disables all forward packages that depend on the given packages
#

MACRO(TRIBITS_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES PACKAGE_NAME)

  #MESSAGE("TRIBITS_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES: ${PACKAGE_NAME}")

  IF ("${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}" STREQUAL "OFF")

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME} TRUE)
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES})
      TRIBITS_PRIVATE_DISABLE_OPTIONAL_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_DISABLE_REQUIRED_PACKAGE_ENABLES(${FWD_DEP_PKG} ${PACKAGE_NAME} FALSE)
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that prints out dependencies for a package
#
# Does not modify the global state.
#

MACRO(TRIBITS_PRINT_PACKAGE_DEPENDENCIES PACKAGE_NAME)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FULL_EXPORT_DEP_PACKAGES)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES)

  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS)
  PRINT_NONEMPTY_VAR(${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS)

ENDMACRO()


#
# Private helper macros
#


MACRO(TRIBITS_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE PACKAGE_NAME  OPTIONAL_DEP_PACKAGE
  TYPE  SET_AS_CACHE_IN
  )

  #MESSAGE("\nPACKAGE_ARCH_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE}")

  IF (SET_AS_CACHE_IN)

    MULTILINE_SET(DOCSTR
      "Enable optional ${TYPE} support in the package ${PACKAGE_NAME}"
      " for the package ${OPTIONAL_DEP_PACKAGE}."
      "  Set to 'ON', 'OFF', or leave empty"
      " to allow for other logic to decide."
      )
  
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} ""
      ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
      SET( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} "" )
    ENDIF()

  ENDIF()

ENDMACRO()


MACRO(TRIBITS_PRIVATE_ADD_OPTIONAL_TPL_ENABLE PACKAGE_NAME OPTIONAL_DEP_TPL
  TYPE  SET_AS_CACHE_IN )

  IF (SET_AS_CACHE_IN)
  
    MULTILINE_SET(DOCSTR
      "Enable optional ${TYPE} support in the package ${PACKAGE_NAME}"
      " for the TPL ${OPTIONAL_DEP_TPL}."
      "  Set to 'ON', 'OFF', or leave empty"
      " to allow for other logic to decide."
      )
    
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} ""
      ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL})
      SET( ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} "" )
    ENDIF()

  ENDIF()

ENDMACRO()


#
# Macro that enables optional package interdependencies
#

MACRO(TRIBITS_ADD_OPTIONAL_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ADD_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  SET(SET_AS_CACHE ${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}})

  IF (SET_AS_CACHE)

    MULTILINE_SET(DOCSTR
      "Build tests for the package ${PACKAGE_NAME}.  Set to 'ON', 'OFF', or leave empty ''"
       " to allow for other logic to decide."
       )
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_TESTS "" ${DOCSTR} )
    
    MULTILINE_SET(DOCSTR
      "Build examples for the package ${PACKAGE_NAME}.  Set to 'ON', 'OFF', or leave empty ''"
       " to allow for other logic to decide."
       )
    SET_CACHE_ON_OFF_EMPTY( ${PACKAGE_NAME}_ENABLE_EXAMPLES "" ${DOCSTR} )

  ELSE()

    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_TESTS)
      SET( ${PACKAGE_NAME}_ENABLE_TESTS "" )
    ENDIF()
    IF (NOT DEFINED ${PACKAGE_NAME}_ENABLE_EXAMPLES)
      SET( ${PACKAGE_NAME}_ENABLE_EXAMPLES "" )
    ENDIF()

  ENDIF() 
 
  FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
    TRIBITS_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} "library" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
    TRIBITS_PRIVATE_ADD_OPTIONAL_PACKAGE_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} "test" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
    TRIBITS_PRIVATE_ADD_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} "library" "${SET_AS_CACHE}" )
  ENDFOREACH()
  
  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
    TRIBITS_PRIVATE_ADD_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} "test" "${SET_AS_CACHE}" )
  ENDFOREACH()

ENDMACRO()


#
# Private helper macros
#

MACRO(TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE PACKAGE_NAME OPTIONAL_DEP_PACKAGE)

  #MESSAGE("TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE}")

  IF("${${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}}" STREQUAL "")
    IF(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} AND ${PROJECT_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
      MESSAGE("-- " "Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}=ON"
       " since ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON AND"
       " ${PROJECT_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE}=ON")
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE} ON)
    ENDIF()
  ENDIF()

  STRING(TOUPPER ${PACKAGE_NAME} PACKAGE_NAME_UPPER)
  STRING(TOUPPER ${OPTIONAL_DEP_PACKAGE} OPTIONAL_DEP_PACKAGE_UPPER)
  SET(MACRO_DEFINE_NAME HAVE_${PACKAGE_NAME_UPPER}_${OPTIONAL_DEP_PACKAGE_UPPER})

  IF(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_PACKAGE})
    SET(${MACRO_DEFINE_NAME} ON)
  ELSE()
    SET(${MACRO_DEFINE_NAME} OFF)
  ENDIF()

ENDMACRO()


MACRO(TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE PACKAGE_NAME OPTIONAL_DEP_TPL)

  #MESSAGE("TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE: ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL}")

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    IF (TPL_ENABLE_${OPTIONAL_DEP_TPL} STREQUAL OFF
      AND ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}
      )
      MESSAGE(
        "\n***"
        "\n*** WARNING: Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}=OFF"
        " which was ON since TPL_ENABLE_${OPTIONAL_DEP_TPL}=OFF"
        "\n***\n"
        )
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} OFF)
    ELSEIF ("${${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}}" STREQUAL ""
      AND TPL_ENABLE_${OPTIONAL_DEP_TPL}
      )
      MESSAGE("-- " "Setting ${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL}=ON"
        " since TPL_ENABLE_${OPTIONAL_DEP_TPL}=ON")
      SET(${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL} ON)
    ENDIF()
  
    STRING(TOUPPER ${PACKAGE_NAME} PACKAGE_NAME_UPPER)
    STRING(TOUPPER ${OPTIONAL_DEP_TPL} OPTIONAL_DEP_TPL_UPPER)
    SET(MACRO_DEFINE_NAME HAVE_${PACKAGE_NAME_UPPER}_${OPTIONAL_DEP_TPL_UPPER})
  
    IF (${PACKAGE_NAME}_ENABLE_${OPTIONAL_DEP_TPL})
      SET(${MACRO_DEFINE_NAME} ON)
    ELSE()
      SET(${MACRO_DEFINE_NAME} OFF)
    ENDIF()

  ENDIF()

ENDMACRO()


#
# Macro that post-processes optional dependancies after all other
# dependencies have been worked out
#

MACRO(TRIBITS_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
      TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} )
    ENDFOREACH()
  
    FOREACH(OPTIONAL_DEP_PACKAGE ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
      TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_PACKAGE_ENABLE(
        ${PACKAGE_NAME} ${OPTIONAL_DEP_PACKAGE} )
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that post-processes final package enables for packages with subpackage
# enables. 
#

MACRO(TRIBITS_POSTPROCESS_PACKAGE_WITH_SUBPACKAGES_ENABLES  PACKAGE_NAME)
  #MESSAGE("TRIBITS_POSTPROCESS_PACKAGE_WITH_SUBPACKAGES_ENABLES  '${PACKAGE_NAME}'")
  FOREACH(TRIBITS_SUBPACKAGE ${${PACKAGE_NAME}_SUBPACKAGES})
    SET(SUBPACKAGE_FULLNAME ${PACKAGE_NAME}${TRIBITS_SUBPACKAGE})
    #PRINT_VAR(${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME})
    #PRINT_VAR(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
    #PRINT_VAR(${SUBPACKAGE_FULLNAME}_ENABLE_TESTS)
    #PRINT_VAR(${PACKAGE_NAME}_ENABLE_TESTS)
    #PRINT_VAR(${SUBPACKAGE_FULLNAME}_ENABLE_EXAMPLES)
    #PRINT_VAR(${PACKAGE_NAME}_ENABLE_EXAMPLES)
    IF (${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME}
      AND "${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}" STREQUAL ""
      )
      MESSAGE("-- "
        "Setting ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON"
        " because ${PROJECT_NAME}_ENABLE_${SUBPACKAGE_FULLNAME}=ON")
      SET(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} ON)
      IF (${SUBPACKAGE_FULLNAME}_ENABLE_TESTS
        AND "${${PACKAGE_NAME}_ENABLE_TESTS}" STREQUAL ""
        )
        MESSAGE("-- "
          "Setting ${PACKAGE_NAME}_ENABLE_TESTS=ON"
          " because ${SUBPACKAGE_FULLNAME}_ENABLE_TESTS=ON")
        SET(${PACKAGE_NAME}_ENABLE_TESTS ON)
      ENDIF()
      IF (${SUBPACKAGE_FULLNAME}_ENABLE_EXAMPLES
        AND "${${PACKAGE_NAME}_ENABLE_EXAMPLES}" STREQUAL ""
        )
        MESSAGE("-- "
          "Setting ${PACKAGE_NAME}_ENABLE_EXAMPLES=ON"
          " because ${SUBPACKAGE_FULLNAME}_ENABLE_EXAMPLES=ON")
        SET(${PACKAGE_NAME}_ENABLE_EXAMPLES ON)
      ENDIF()
    ENDIF()
  ENDFOREACH()
ENDMACRO()


#
# Macro that post-processes optional package TPL based on if the TPL
# has been enabled or not
#

MACRO(TRIBITS_POSTPROCESS_OPTIONAL_TPL_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ADD_OPTIONAL_TPL_ENABLES: ${PACKAGE_NAME}")

  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
    TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} )
  ENDFOREACH()

  FOREACH(OPTIONAL_DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
    TRIBITS_PRIVATE_POSTPROCESS_OPTIONAL_TPL_ENABLE(
      ${PACKAGE_NAME} ${OPTIONAL_DEP_TPL} )
  ENDFOREACH()

ENDMACRO()


#
# Set an individual package variable enable based on the global value
#

MACRO(TRIBITS_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE PACKAGE_ARCH_VAR PACKAGE_VAR)

  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    MESSAGE("")
    MESSAGE("TRIBITS_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE:")
    MESSAGE("-- " "${PACKAGE_ARCH_VAR} = ${${PACKAGE_ARCH_VAR}}")
    MESSAGE("-- " "${PACKAGE_VAR} = ${${PACKAGE_VAR}}")
  ENDIF()

  IF ("${${PACKAGE_VAR}}" STREQUAL "")
    IF (${PACKAGE_ARCH_VAR} STREQUAL "ON")
      MESSAGE("-- " "Setting ${PACKAGE_VAR}=ON")
      SET(${PACKAGE_VAR} ON)
    ELSEIF (PACKAGE_ARCH_VAR STREQUAL "OFF")
      MESSAGE("-- " "Setting ${PACKAGE_VAR}=OFF")
      SET(${PACKAGE_VAR} OFF)
    ELSE()
      IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
        MESSAGE("-- " "ELSE")
        # Otherwise, we will leave it up the the individual package
        # to decide?
      ENDIF()
    ENDIF()
  ELSE()
    IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
      MESSAGE("-- " "${PACKAGE_VAR} NOT DEFAULT")
    ENDIF()
  ENDIF()

  IF (${PROJECT_NAME}_VERBOSE_CONFIGURE)
    MESSAGE("-- " "${PACKAGE_VAR} = ${${PACKAGE_VAR}}")
  ENDIF()

ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} based on
# ${PROJECT_NAME}_ENABLE_ALL_PACKAGES
#

MACRO(TRIBITS_APPLY_ALL_PACKAGE_ENABLES PACKAGE_NAME)
  TRIBITS_IMPLICIT_PACKAGE_ENABLE_IS_ALLOWED( "" ${PACKAGE_NAME}
    PROCESS_PACKAGE_ENABLE )
  IF (PROCESS_PACKAGE_ENABLE)
    TRIBITS_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_ALL_PACKAGES ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME} )
  ENDIF()
ENDMACRO()


#
# Macro used to set ${TRIBITS_PACKAGE)_ENABLE_TESTS and ${TRIBITS_PACKAGE)_ENABLE_EXAMPLES
# based on ${PROJECT_NAME}_ENABLE_ALL_PACKAGES
#

MACRO(TRIBITS_APPLY_TEST_EXAMPLE_ENABLES PACKAGE_NAME)
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
    TRIBITS_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_TESTS ${PACKAGE_NAME}_ENABLE_TESTS )
    TRIBITS_SET_ALL_PACKAGES_PACKAGE_ENABLE_VARIABLE(
      ${PROJECT_NAME}_ENABLE_EXAMPLES ${PACKAGE_NAME}_ENABLE_EXAMPLES )
  ENDIF()
ENDMACRO()


#
# Private helper macro
#

MACRO(TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE  FORWARD_DEP_PACKAGE_NAME  PACKAGE_NAME)
  TRIBITS_IMPLICIT_PACKAGE_ENABLE_IS_ALLOWED( "" ${FORWARD_DEP_PACKAGE_NAME}
    ALLOW_PACKAGE_ENABLE )
  #MESSAGE("TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE: "
  #  "${FORWARD_DEP_PACKAGE_NAME} ${PACKAGE_NAME} ${ALLOW_PACKAGE_ENABLE}") 
  # Enable the forward package if it is not already set to ON or OFF
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
  IF(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} STREQUAL ""
    AND ALLOW_PACKAGE_ENABLE
    )
    MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME}=ON"
      " because ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON")
    ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME})
    SET(${PROJECT_NAME}_ENABLE_${FORWARD_DEP_PACKAGE_NAME} ON)
  ENDIF()
ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${FWD_PACKAGE_NAME)=ON for all optional
# and required forward library dependencies of the package ${PACKAGE_NAME}
#

MACRO(TRIBITS_ENABLE_FORWARD_LIB_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ENABLE_FORWARD_PACKAGE_ENABLES ${PACKAGE_NAME}")
  #PRINT_VAR(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  # Enable the forward packages if this package is enabled
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_LIB_OPTIONAL_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro used to set ${PROJECT_NAME}_ENABLE_${FWD_PACKAGE_NAME)=ON for all optional
# and required forward test/example dependencies of the package ${PACKAGE_NAME}
#

MACRO(TRIBITS_ENABLE_FORWARD_TEST_PACKAGE_ENABLES PACKAGE_NAME)

  #MESSAGE("\nPACKAGE_ARCH_ENABLE_FORWARD_PACKAGE_ENABLES ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  # Enable the forward packages if this package is enabled
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

    FOREACH(FWD_DEP_PKG ${${PACKAGE_NAME}_FORWARD_TEST_OPTIONAL_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_FORWARD_PACKAGE(${FWD_DEP_PKG} ${PACKAGE_NAME})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Private helper macros
#

MACRO(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGE  PACKAGE_NAME  DEP_PACKAGE_NAME)

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_SECONDARY_STABLE_CODE)
  #PRINT_VAR(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE)
  #PRINT_VAR(${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} STREQUAL "")

    SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_ENABLE_PACKAGE "")

    # Enable the package if there is an optional dependence and we are asked
    # to enabled optional dependencies.
    IF (TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE)
      TRIBITS_IMPLICIT_PACKAGE_ENABLE_IS_ALLOWED(${PACKAGE_NAME} ${DEP_PACKAGE_NAME}
        ALLOW_IMPLICIT_ENABLE)
      IF (ALLOW_IMPLICIT_ENABLE)
        MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
          " because ${PACKAGE_NAME} has an optional dependence on ${DEP_PACKAGE_NAME}")
        SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_ENABLE_PACKAGE ON)
      ENDIF()
    ENDIF()

    # Enable the package if the user directly specified the optional package
    # enable reguardless if it is PS or SS or even EX.
    IF (${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})
      MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
        " because ${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON")
      SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_ENABLE_PACKAGE ON)
    ENDIF()

    IF (TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_ENABLE_PACKAGE)
      ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
      SET(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} ON)
    ENDIF()

  ENDIF()

ENDMACRO()


MACRO(TRIBITS_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE  PACKAGE_NAME  DEP_PACKAGE_NAME)
  #MESSAGE("TRIBITS_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE: ${PACKAGE_NAME} ${DEP_PACKAGE_NAME}")
  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  #PRINT_VAR(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE)
  #PRINT_VAR(${PACKAGE_NAME}_ENABLE_${DEP_PACKAGE_NAME})
  IF (${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} STREQUAL "")
    MESSAGE("-- " "Setting ${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME}=ON"
      " because ${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=ON")
    ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME})
    SET(${PROJECT_NAME}_ENABLE_${DEP_PACKAGE_NAME} ON)
  ENDIF()
ENDMACRO()


MACRO(TRIBITS_PRIVATE_ENABLE_DEP_TPL PACKAGE_NAME DEP_TPL_NAME)
  ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
  IF(TPL_ENABLE_${DEP_TPL_NAME} STREQUAL "")
    MESSAGE("-- " "Setting TPL_ENABLE_${DEP_TPL_NAME}=ON because"
      " it is required by the enabled package ${PACKAGE_NAME}")
    ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
    SET(TPL_ENABLE_${DEP_TPL_NAME} ON)
  ENDIF()
ENDMACRO()


MACRO(TRIBITS_PRIVATE_ENABLE_OPTIONAL_DEP_TPL PACKAGE_NAME DEP_TPL_NAME)
  #ASSERT_DEFINED(${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME})
  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}
    AND ${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME}
    AND TPL_ENABLE_${DEP_TPL_NAME} STREQUAL ""
    )
    MESSAGE("-- " "Setting TPL_ENABLE_${DEP_TPL_NAME}=ON because"
      " ${PACKAGE_NAME}_ENABLE_${DEP_TPL_NAME}=ON")
    ASSERT_DEFINED(TPL_ENABLE_${DEP_TPL_NAME})
    SET(TPL_ENABLE_${DEP_TPL_NAME} ON)
  ENDIF()
ENDMACRO()


#
# Macro that enables the optional TPLs for given package
#

MACRO(TRIBITS_ENABLE_OPTIONAL_TPLS PACKAGE_NAME)

  #MESSAGE("TRIBITS_ENABLE_OPTIONAL_TPLS: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_TPLS})
      TRIBITS_PRIVATE_ENABLE_OPTIONAL_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_TPLS})
      TRIBITS_PRIVATE_ENABLE_OPTIONAL_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that sets the optional packages for given package
#
# Here I have to enable the required packages too or the logic just does no
# work as expected.
#

MACRO(TRIBITS_ENABLE_OPTIONAL_PACKAGES PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_OPTIONAL_PACKAGE_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  #PRINT_VAR(TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE)

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES})
      SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE
        ${TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE})
      TRIBITS_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_OPTIONAL_DEP_PACKAGES})
      SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE
       ${TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE})
      TRIBITS_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES})
      SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE
        ${TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE})
      TRIBITS_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_OPTIONAL_DEP_PACKAGES})
      SET(TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE
        ${TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE})
      TRIBITS_PRIVATE_ENABLE_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG})
    ENDFOREACH()

    # 2008/02/13: rabartl: Above, I had to set the varaible
    # TRIBITS_PRIVATE_ENABLE_DEP_PACKAGES_IMPLICIT_ENABLE and then call the
    # function because it was not being passed as an argument as was zero on
    # the other side.  This is very strange but you have to do what you have
    # to do!

  ENDIF()

ENDMACRO()


#
# Macro that sets the required packages for given package
#

MACRO(TRIBITS_ENABLE_REQUIRED_PACKAGES PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_REQUIRED_PACKAGE_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG} ON)
    ENDFOREACH()

    FOREACH(DEP_PKG ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_PACKAGES})
      TRIBITS_PRIVATE_ENABLE_REQUIRED_DEP_PACKAGE(${PACKAGE_NAME} ${DEP_PKG} ON)
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that sets the required TPLs for given package
#

MACRO(TRIBITS_ENABLE_REQUIRED_TPLS PACKAGE_NAME)

  #MESSAGE("PACKAGE_ARCH_ENABLE_REQUIRED_TPL_ENABLES: ${PACKAGE_NAME}")
  #MESSAGE("-- " "${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}=${${PROJECT_NAME}_ENABLE_${PACKAGE_NAME}}")

  ASSERT_DEFINED(${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

  IF (${PROJECT_NAME}_ENABLE_${PACKAGE_NAME})

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_LIB_REQUIRED_DEP_TPLS})
      TRIBITS_PRIVATE_ENABLE_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

    FOREACH(DEP_TPL ${${PACKAGE_NAME}_TEST_REQUIRED_DEP_TPLS})
      TRIBITS_PRIVATE_ENABLE_DEP_TPL(${PACKAGE_NAME} ${DEP_TPL})
    ENDFOREACH()

  ENDIF()

ENDMACRO()


#
# Macro that adjusts all of the package enables from what the user input
# to the final set that will be used to enable packages
#

MACRO(TRIBITS_ADJUST_PACKAGE_ENABLES)

  IF (${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES)
    MESSAGE("")
    MESSAGE("Setting to empty '' all enabled packages on reqeust ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
      IF (${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE})
        SET_CACHE_ON_OFF_EMPTY(${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE} ""
          "Forced to empty '' by ${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES=ON" FORCE)
        SET(${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE} "")
      ENDIF()
      #PRINT_VAR(${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE})
      # NOTE: Above, we don't want to set to empty those packages that have hard
      # disables because this will mess up the logic in later invocations.
    ENDFOREACH()
    ADVANCED_SET(${PROJECT_NAME}_UNENABLE_ENABLED_PACKAGES OFF CACHE BOOL
      "Forced to FALSE after use" FORCE)
  ENDIF()

  MESSAGE("")
  MESSAGE("Disabling all packages that have a required dependency"
    " on disabled TPLs and optional package TPL support based on TPL_ENABLE_<TPL>=OFF ...")
  MESSAGE("")
  FOREACH(TPL ${${PROJECT_NAME}_TPLS})
    TRIBITS_DISABLE_TPL_DEP_PACKAGES(${TPL})
  ENDFOREACH()

  MESSAGE("")
  MESSAGE("Disabling subpackages for hard disables of parent packages"
    " ${PROJECT_NAME}_ENABLE_<PARENT_PACKAGE>=OFF ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_PACKAGES})
    TRIBITS_DISABLE_PARENTS_SUBPACKAGES(${TRIBITS_PACKAGE})
  ENDFOREACH()

  MESSAGE("")
  MESSAGE("Disabling forward required packages and optional intra-package"
    " support that have a dependancy on disabled packages"
    " ${PROJECT_NAME}_ENABLE_<TRIBITS_PACKAGE>=OFF ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_DISABLE_FORWARD_REQUIRED_DEP_PACKAGES(${TRIBITS_PACKAGE})
  ENDFOREACH()
  
  IF (${PROJECT_NAME}_ENABLE_ALL_PACKAGES)
    MESSAGE("")
    MESSAGE("Enabling all packages that are not currently disabled because of"
      " ${PROJECT_NAME}_ENABLE_ALL_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_PACKAGES})
      TRIBITS_APPLY_ALL_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
    ENDFOREACH()
  ENDIF()
  
  IF (${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES)
    MESSAGE("")
    MESSAGE("Sweep forward enabling all forward library dependent packages because"
      " ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
      TRIBITS_ENABLE_FORWARD_LIB_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
    ENDFOREACH()
    MESSAGE("")
    MESSAGE("Sweep backward enabling all forward test dependent packages because"
      " ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_REVERSE_SE_PACKAGES})
      TRIBITS_ENABLE_FORWARD_TEST_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
    ENDFOREACH()
    # NOTE: Above, we want to sweep backward to enable test-dependent packages
    # because we don't want to enable pacakge Z just because pacakge Y was enabled
    # because it had a test-only dependency on package X.  Sweeping backwards through
    # the packages makes sure this does not happen.
    SET(${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES ON)
  ENDIF()
  
  IF (${PROJECT_NAME}_ENABLE_TESTS OR ${PROJECT_NAME}_ENABLE_EXAMPLES)
    MESSAGE("")
    MESSAGE("Enabling all tests and/or examples that have not been"
      " explicitly disabled because ${PROJECT_NAME}_ENABLE_[TESTS,EXAMPLES]=ON ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
      TRIBITS_APPLY_TEST_EXAMPLE_ENABLES(${TRIBITS_PACKAGE})
    ENDFOREACH()
  ENDIF()
  # NOTE: Above, we enable tests and examples here, before the remaining required
  # packages so that we don't enable tests that don't need to be enabled based
  # on the use of the option ${PROJECT_NAME}_ENABLE_ALL_FORWARD_DEP_PACKAGES.

  IF (${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES)
    MESSAGE("")
    MESSAGE("Enabling all optional packages for current set of enabled"
      " packages because ${PROJECT_NAME}_ENABLE_ALL_OPTIONAL_PACKAGES=ON ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_REVERSE_SE_PACKAGES})
      SET(TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE ON) # Hack to get around CMake bug?
      TRIBITS_ENABLE_OPTIONAL_PACKAGES(${TRIBITS_PACKAGE})
    ENDFOREACH()
    # NOTE: Above, we have to loop through the packages backward to enable all the
    # packages that feed into these packages.
    # NOTE Above, we don't have to enable the required packages because that will
    # come next
  ELSE()
    MESSAGE("")
    MESSAGE("Enabling all packages that are not disabled based on"
      " <TRIBITS_PACKAGE>_ENABLE_<DEPPACKAGE>=ON  ...")
    MESSAGE("")
    FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_REVERSE_SE_PACKAGES})
      SET(TRIBITS_ENABLE_OPTIONAL_PACKAGES_IMPLICIT_ENABLE OFF) # Hack to get around CMake bug?
      TRIBITS_ENABLE_OPTIONAL_PACKAGES(${TRIBITS_PACKAGE})
    ENDFOREACH()
    # NOTE: Aaove, we loop backwards through the packages to pick up as many
    # package enables as we can.  Also, we do this after the tests/examples are 
    # enabled.
  ENDIF()
  
  MESSAGE("")
  MESSAGE("Enabling all remaining required packages for the current set"
    " of enabled packages ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_REVERSE_SE_PACKAGES})
    TRIBITS_ENABLE_REQUIRED_PACKAGES(${TRIBITS_PACKAGE})
  ENDFOREACH()
  
  MESSAGE("")
  MESSAGE("Enabling all optional intra-package enables <TRIBITS_PACKAGE>_ENABLE_<DEPPACKAGE>"
    " that are not currently disabled if both sets of packages are enabled ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_POSTPROCESS_OPTIONAL_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
  ENDFOREACH()

  MESSAGE("")
  MESSAGE("Enabling all remaining required TPLs for current set of"
    " enabled packages ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_ENABLE_REQUIRED_TPLS(${TRIBITS_PACKAGE})
  ENDFOREACH()
  
  MESSAGE("")
  MESSAGE("Enabling all optional package TPL support for currently"
    " enabled TPLs ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_POSTPROCESS_OPTIONAL_TPL_ENABLES(${TRIBITS_PACKAGE})
  ENDFOREACH()
  
  MESSAGE("")
  MESSAGE("Enabling TPLs based on <TRIBITS_PACKAGE>_ENABLE_<TPL>=ON if TPL is not explicitly disabled ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_ENABLE_OPTIONAL_TPLS(${TRIBITS_PACKAGE})
  ENDFOREACH()
  # NOTE: We need to do this after the above optional package TPL support
  # logic so that the TPL will be turned on for this package only as requested
  # in bug 4298.

  MESSAGE("")
  MESSAGE("Set cache entries for optional packages/TPLs and tests/examples for packages actually enabled ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_SE_PACKAGES})
    TRIBITS_ADD_OPTIONAL_PACKAGE_ENABLES(${TRIBITS_PACKAGE})
  ENDFOREACH()
  
  MESSAGE("")
  MESSAGE("Enabling all packages not yet enabled that have at least one subpackage enabled ...")
  MESSAGE("")
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_PACKAGES})
    TRIBITS_POSTPROCESS_PACKAGE_WITH_SUBPACKAGES_ENABLES(${TRIBITS_PACKAGE})
  ENDFOREACH()

ENDMACRO()


#
# Macro that post-modifies the list of enabled packages to disable packages
# that are to be excluded based on the Repository explicit include list.
#
# This function is used in TribitsCTestDriverCore.cmake to exclude packages
# implicitly disabled a repository given by certain logic.  Based on this
# logic, we don't want to disable packages that were explicitly turned on an
# have tests enabled.
#

MACRO(TRIBITS_APPLY_REPOSITORY_NO_IMPLICIT_PACKAGE_ENABLE_DISABLE)
  FOREACH(TRIBITS_PACKAGE ${${PROJECT_NAME}_PACKAGES})
    IF (${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE} AND NOT ${TRIBITS_PACKAGE}_ENABLE_TESTS)
      TRIBITS_REPOSITORY_IMLICIT_PACKAGE_ENABLE_IS_ALLOWED("" ${TRIBITS_PACKAGE}
        ALLOW_IMPLICIT_ENABLE
        )
      IF (NOT ALLOW_IMPLICIT_ENABLE)
        SET(${PROJECT_NAME}_ENABLE_${TRIBITS_PACKAGE} OFF)
      ENDIF()
    ENDIF()
  ENDFOREACH()
ENDMACRO()
