# @HEADER
# ************************************************************************
#
#            TriBITS: Tribial Build, Integrate, and Test System
#                    Copyright 2013 Sandia Corporation
#
# Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
# the U.S. Government retains certain rights in this software.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. Neither the name of the Corporation nor the names of the
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# ************************************************************************
# @HEADER

MESSAGE("PROJECT_NAME = ${PROJECT_NAME}")
MESSAGE("${PROJECT_NAME}_TRIBITS_DIR = ${${PROJECT_NAME}_TRIBITS_DIR}")

SET( CMAKE_MODULE_PATH
  "${${PROJECT_NAME}_TRIBITS_DIR}/utils"
  "${${PROJECT_NAME}_TRIBITS_DIR}/package_arch"
  )

SET(PACKAGE_ADD_EXECUTABLE_UNIT_TESTING ON)

INCLUDE(MessageWrapper)
INCLUDE(TribitsTestCategories)
INCLUDE(TribitsAddTest)
INCLUDE(TribitsAddAdvancedTest)
INCLUDE(TribitsAddExecutableAndTest)
INCLUDE(TribitsETISupport)
INCLUDE(UnitTestHelpers)
INCLUDE(GlobalSet)
INCLUDE(GlobalNullSet)
INCLUDE(AppendStringVar)

#####################################################################
#
# Unit tests for PACKAGE_ADD_XXX(...) CMake commands run as CMake code
#
# This file contains a set of unit tests for the package_arch macros
# PACKAGE_ADD_XXX(...) functions.  These unit tests are written in CMake
# itself.  This is not a very advanced unit testing system and it not that
# easy to work with.  However, it does perform some pretty strong testing and
# is much better than doing nothing.
#
#####################################################################


#
# Set up unit test functions that will be called below to actually run the
# unit tests.
#
# The reason that we use functions is so that we can change varibles just
# inside of the functions that have their own variable scoping.  In that way,
# we can keep variables that are set in one unit test from affecting the
# others.
#


FUNCTION(UNIT_TEST_APPEND_STRING_VAR)

  MESSAGE("\n***")
  MESSAGE("*** Testing APPEND_STRING_VAR()")
  MESSAGE("***\n")

  MESSAGE("APPEND_STRING_VAR(): Testing simple concatenation")
  SET(SOME_STRING_VAR "")
  APPEND_STRING_VAR(SOME_STRING_VAR
     "begin\n" )
  APPEND_STRING_VAR(SOME_STRING_VAR
     "middle1" " middile2" " middle3\n" )
  APPEND_STRING_VAR(SOME_STRING_VAR
     "end\n" )
  UNITTEST_COMPARE_CONST(SOME_STRING_VAR
    "begin\nmiddle1 middile2 middle3\nend\n")

  MESSAGE("APPEND_STRING_VAR(): Testing with [] and {} which messes up ;")
  SET(SOME_STRING_VAR "")
  APPEND_STRING_VAR(SOME_STRING_VAR
     "[\n" )
  APPEND_STRING_VAR(SOME_STRING_VAR
     "{middle1" " middile2" " middle3}\n" )
  APPEND_STRING_VAR(SOME_STRING_VAR
     "]\n" )
  UNITTEST_COMPARE_CONST(SOME_STRING_VAR
    "[\n;{middle1; middile2; middle3}\n;]\n")

  MESSAGE("APPEND_STRING_VAR_EXT(): Testing with [] and {} which ignores ;")
  SET(SOME_STRING_VAR "")
  APPEND_STRING_VAR_EXT(SOME_STRING_VAR
     "[\n" )
  APPEND_STRING_VAR_EXT(SOME_STRING_VAR
     "{middle1 middile2 middle3}\n" )
  APPEND_STRING_VAR_EXT(SOME_STRING_VAR
     "]\n" )
  UNITTEST_COMPARE_CONST(SOME_STRING_VAR
    "[\n{middle1 middile2 middle3}\n]\n")

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ARCH_MISC)

  MESSAGE("\n***")
  MESSAGE("*** Testing miscellaneous TriBITS macro functionality")
  MESSAGE("***\n")

  MESSAGE("Testing MESSAGE_WRAPPER(...) without capture")
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE FALSE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT "Dummy")
  MESSAGE_WRAPPER("Some message that should get printed and not intercepted")
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT "Dummy")

  MESSAGE("Testing MESSAGE_WRAPPER(...) with capture")
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE TRUE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT "Dummy")
  MESSAGE_WRAPPER("Some message that should get intercepted")
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT "Dummy;Some message that should get intercepted")

  MESSAGE("Testing FIND_LIST_ELEMENT(${PROJECT_NAME}_VALID_CATEGORIES BASIC ELEMENT_FOUND)")
  FIND_LIST_ELEMENT(${PROJECT_NAME}_VALID_CATEGORIES BASIC ELEMENT_FOUND)
  UNITTEST_COMPARE_CONST(ELEMENT_FOUND "TRUE")

  MESSAGE("Testing FIND_LIST_ELEMENT(${PROJECT_NAME}_VALID_CATEGORIES BADCAT ELEMENT_FOUND)")
  FIND_LIST_ELEMENT(${PROJECT_NAME}_VALID_CATEGORIES BADCAT ELEMENT_FOUND)
  UNITTEST_COMPARE_CONST(ELEMENT_FOUND "FALSE")

  MESSAGE("Testing TRIBITS_GET_INVALID_CATEGORIES( ... BADCAT)")
  TRIBITS_GET_INVALID_CATEGORIES(INVALID_CATEGORIES BADCAT)  
  UNITTEST_COMPARE_CONST( INVALID_CATEGORIES "BADCAT" )

  MESSAGE("Testing TRIBITS_GET_INVALID_CATEGORIES( ... BADCAT BASIC)")
  TRIBITS_GET_INVALID_CATEGORIES(INVALID_CATEGORIES BADCAT BASIC)  
  UNITTEST_COMPARE_CONST( INVALID_CATEGORIES "BADCAT" )

  MESSAGE("Testing TRIBITS_GET_INVALID_CATEGORIES( ... BASIC BADCAT)")
  TRIBITS_GET_INVALID_CATEGORIES(INVALID_CATEGORIES BASIC BADCAT)  
  UNITTEST_COMPARE_CONST( INVALID_CATEGORIES "BADCAT" )

  MESSAGE("Testing TRIBITS_GET_INVALID_CATEGORIES( ... BADCAT1 BADCAT2)")
  TRIBITS_GET_INVALID_CATEGORIES(INVALID_CATEGORIES BADCAT1 BADCAT2)  
  UNITTEST_COMPARE_CONST( INVALID_CATEGORIES "BADCAT1;BADCAT2" )

  MESSAGE("Testing TRIBITS_GET_INVALID_CATEGORIES( ... BASIC BADCAT1 NIGHTLY BADCAT2 PERFORMANCE)")
  TRIBITS_GET_INVALID_CATEGORIES(INVALID_CATEGORIES BASIC BADCAT1 NIGHTLY BADCAT2 PERFORMANCE)  
  UNITTEST_COMPARE_CONST( INVALID_CATEGORIES "BADCAT1;BADCAT2" )

  MESSAGE("Testing TRIBITS_ASSERT_VALID_CATEGORIES( ... BADCAT1 BASIC BADCAT2)")
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE TRUE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT)
  TRIBITS_ASSERT_VALID_CATEGORIES(BADCAT1 BASIC BADCAT2)  
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE FALSE)
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT
    "SEND_ERROR;Error: The categories 'BADCAT1;BADCAT2' are not; in the list of valid categories '${${PROJECT_NAME}_VALID_CATEGORIES_STR}'!")

  MESSAGE("Testing TRIBITS_ASSERT_VALID_CATEGORIES( ... BASIC)")
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE TRUE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT "Dummy")
  TRIBITS_ASSERT_VALID_CATEGORIES(BASIC)  
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE FALSE)
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT "Dummy")

  MESSAGE("Testing TRIBITS_ASSERT_VALID_CATEGORIES( ... BASIC NIGHTLY)")
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE TRUE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT "Dummy")
  TRIBITS_ASSERT_VALID_CATEGORIES(BASIC NIGHTLY)  
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE FALSE)
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT "Dummy")

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_TEST_BASIC)

  MESSAGE("\n***")
  MESSAGE("*** Testing basic functionality of TRIBITS_ADD_TEST(...)")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageA)

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN})

  MESSAGE("Test in HOST")
  SET(${PROJECT_NAME}_HOSTNAME MyHost)
  TRIBITS_ADD_TEST( ${EXEN} HOST MyHost )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test not in HOST")
  SET(${PROJECT_NAME}_HOSTNAME TheHost)
  TRIBITS_ADD_TEST( ${EXEN} HOST MyHost )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test in XHOST")
  SET(${PROJECT_NAME}_HOSTNAME MyHost)
  TRIBITS_ADD_TEST( ${EXEN} XHOST MyHost )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test not in XHOST")
  SET(${PROJECT_NAME}_HOSTNAME TheHost)
  TRIBITS_ADD_TEST( ${EXEN} XHOST MyHost )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test in HOSTTYPE")
  SET(CMAKE_HOST_SYSTEM_NAME MyHostType)
  TRIBITS_ADD_TEST( ${EXEN} HOSTTYPE MyHostType )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test not in HOSTTYPE")
  SET(CMAKE_HOST_SYSTEM_NAME TheHostType)
  TRIBITS_ADD_TEST( ${EXEN} HOSTTYPE MyHostType )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test in XHOSTTYPE")
  SET(CMAKE_HOST_SYSTEM_NAME MyHostType)
  TRIBITS_ADD_TEST( ${EXEN} XHOSTTYPE MyHostType )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test not in XHOSTTYPE")
  SET(CMAKE_HOST_SYSTEM_NAME TheHostType)
  TRIBITS_ADD_TEST( ${EXEN} XHOSTTYPE MyHostType )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a single basic test with no arguments")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a single basic test with a single argument")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1"
    )

  MESSAGE("Add a single basic test with two arguments")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add two tests with simple arguments")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1" "arg2 arg3" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_0;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;${PACKEXEN}_1;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg2;arg3"
    )

  MESSAGE("Add a double quoted input argument")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "--arg1=\"bob and cats\"" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;--arg1=\"bob and cats\""
    )

  MESSAGE("Add a double quoted with single quotes input argument")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "--arg1=\"'bob' and 'cats'\"" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;--arg1=\"'bob' and 'cats'\""
    )

  MESSAGE("Add two tests with different postfixes and arguments")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN}
    POSTFIX_AND_ARGS_0 pf_arg1 arg1
    POSTFIX_AND_ARGS_1 pf_arg23 arg2 arg3
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_pf_arg1;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;${PACKEXEN}_pf_arg23;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg2;arg3"
    )

  MESSAGE("Add an executable with no prefix")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NOEXEPREFIX ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${EXEN}.exe;arg1"
    )

  MESSAGE("Add an executable with no suffix")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NOEXESUFFIX ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN};arg1"
    )

  MESSAGE("Add an executable with no prefix and no suffix")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NOEXEPREFIX NOEXESUFFIX ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${EXEN};arg1"
    )

  MESSAGE("Add a test with a different name from the executable")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NAME SomeOtherName ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKAGE_NAME}_SomeOtherName;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1"
    )

  MESSAGE("Add a test with with a postfix appended to the executable name")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NAME_POSTFIX somePostfix ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_somePostfix;${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1"
    )

  MESSAGE("Add a test with the directory overridden")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} DIRECTORY "../somedir" ARGS arg1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/../somedir/${PACKEXEN}.exe;arg1"
    )

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_TEST_DISABLE)

  MESSAGE("\n***")
  MESSAGE("*** Testing test-by-test disable of TRIBITS_ADD_TEST(...)")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageA)

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN})

  MESSAGE("Check that TRIBITS_ADD_TEST(...) adds test")
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Check that PackageA_SomeExec_DISABLE=ON disables TRIBITS_ADD_TEST(...)")
  SET(PackageA_SomeExec_DISABLE ON)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Check that TRIBITS_ADD_ADVANCED_TEST(...) adds test")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( SomeCmnd
    TEST_0 CMND someCmnd
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"someCmnd\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )

  MESSAGE("Check that PackageA_SomeCmnd_DISABLE=ON disables TRIBITS_ADD_ADVANCED_TEST(...)")
  SET(PackageA_SomeCmnd_DISABLE ON)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( SomeCmnd
    TEST_0 CMND someCmnd
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_TEST_CATEGORIES)

  MESSAGE("\n***")
  MESSAGE("*** Testing TRIBITS_ADD_TEST( ... CATEGORIES ... )")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageA)

  SET(${PROJECT_NAME}_TEST_CATEGORIES "")

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN})

  MESSAGE("Test no category matching NIGHTLY category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test no category matching BASIC category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test NIGHTLY category *not* matching BASIC category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_TEST( ${EXEN} CATEGORIES NIGHTLY )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test BASIC category matching BASIC category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_TEST( ${EXEN} CATEGORIES BASIC )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test no category *not* matching PERFORMANCE category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES PERFORMANCE)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Test PERFORMANCE category matching PERFORMANCE category set by client")
  SET(${PROJECT_NAME}_TEST_CATEGORIES PERFORMANCE)
  TRIBITS_ADD_TEST( ${EXEN} CATEGORIES PERFORMANCE )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Test invalid BADCAT category not matching anything and resulting in error")
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE TRUE)
  GLOBAL_SET(MESSAGE_WRAPPER_INPUT "")
  TRIBITS_ADD_TEST( ${EXEN} CATEGORIES BADCAT )
  SET(MESSAGE_WRAPPER_UNIT_TEST_MODE FALSE)
  UNITTEST_COMPARE_CONST(MESSAGE_WRAPPER_INPUT
    "SEND_ERROR;Error: The categories 'BADCAT' are not; in the list of valid categories '${${PROJECT_NAME}_VALID_CATEGORIES_STR}'!")
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "")

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_TEST_COMM)

  MESSAGE("\n***")
  MESSAGE("*** Testing TRIBITS_ADD_TEST( ... COMM ... )")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageB)

  # Doing default serial mode

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN})
  SET(MPI_EXEC_MAX_NUMPROCS 5)
  SET(MPI_EXEC mpiexec)
  SET(MPI_EXEC_PRE_NUMPROCS_FLAGS --pre-num-procs-flags)
  SET(MPI_EXEC_NUMPROCS_FLAG --num-procs)
  SET(MPI_EXEC_POST_NUMPROCS_FLAGS --post-num-procs-flags)

  MESSAGE("Add a test for serial with no COMM input")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a test for MPI with no COMM input")
  SET(TPL_ENABLE_MPI ON)
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  # Doing MPI mode
  SET(TPL_ENABLE_MPI ON)

  MESSAGE("Add a test for MPI with no COMM input but with some args")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add a serial-only in an MPI-only build (adds no test)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} COMM serial )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Add a test for MPI with 'COMM mpi'")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add an MPI test with 2 procs")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi NUM_MPI_PROCS 2 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_2;${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};2;${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add an MPI test with more than the number of allowed processors (will not be added)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi NUM_MPI_PROCS 10 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Add an MPI test with NUM_PROCS 1-10 (will be max num procs)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi NUM_MPI_PROCS 1-10 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add an MPI test with NUM_PROCS 3-10 (will be max num procs)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi NUM_MPI_PROCS 3-10 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  MESSAGE("Add an MPI test with NUM_PROCS ${MPI_EXEC_MAX_NUMPROCS}-10 (will be max num procs)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} ARGS "arg1 arg2" COMM mpi NUM_MPI_PROCS ${MPI_EXEC_MAX_NUMPROCS}-10 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;arg2"
    )

  # Doing serial mode
  SET(TPL_ENABLE_MPI OFF)

  MESSAGE("Add a test for serial mode with 'COMM serial'")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} COMM serial )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a test for serial mode with 'COMM mpi (adds no test)")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} COMM mpi )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    ""
    )

  MESSAGE("Add a test with MPI and NAME_POSTFIX")
  SET(TPL_ENABLE_MPI ON)
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NAME_POSTFIX mypostfix1 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_mypostfix1_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a test with MPI and NAME")
  SET(TPL_ENABLE_MPI ON)
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NAME ${EXEN}_mypostfix2 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_mypostfix2_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe"
    )

  MESSAGE("Add a test with MPI, two arguments, and NAME_POSTFIX")
  SET(TPL_ENABLE_MPI ON)
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_TEST( ${EXEN} NAME_POSTFIX mypostfix3 ARGS "arg1" "arg2" )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_ADD_TEST_INPUT
    "${PACKEXEN}_mypostfix3_0_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg1;${PACKEXEN}_mypostfix3_1_MPI_${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC};${MPI_EXEC_PRE_NUMPROCS_FLAGS};${MPI_EXEC_NUMPROCS_FLAG};${MPI_EXEC_MAX_NUMPROCS};${MPI_EXEC_POST_NUMPROCS_FLAGS};${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe;arg2"
    )

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_ADVANCED_TEST_BASIC)

  MESSAGE("\n***")
  MESSAGE("*** Testing basic functionality of TRIBITS_ADD_ADVANCED_TEST(...)")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_ADVANCED_TEST(...)
  SET(PACKAGE_NAME PackageA)

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN}.exe)
  SET(CMNDN ls)

  MESSAGE("***\n*** Add a single basic command with no arguments\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_cmnd_1_args_0
    TEST_0 CMND ${CMNDN}
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMNDN}\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_cmnd_1_args_0.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"ls\""
      "NUM_CMNDS 1"
    )

  MESSAGE("***\n*** Add a single package executable with no arguments\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_exec_1_args_0
    TEST_0 EXEC ${EXEN}
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_exec_1_args_0.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("***\n*** Add a single basic command with two arguments\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_cmnd_1_args_2
    TEST_0 CMND ${CMNDN} ARGS CMakeLists.txt CMakeFiles
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMNDN}\" \"CMakeLists.txt\" \"CMakeFiles\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_cmnd_1_args_2.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"ls\" \"CMakeLists.txt\" \"CMakeFiles\""
      "NUM_CMNDS 1"
      "CMAKE_MODULE_PATH"
      "DriveAdvancedTest"
      "DRIVE_ADVANCED_TEST"
    )

  MESSAGE("***\n*** Add a single package executable with three arguments\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_exec_1_args_3
    TEST_0 EXEC ${EXEN} ARGS arg1 arg2 arg3
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\" \"arg1\" \"arg2\" \"arg3\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_exec_1_args_3.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\" \"arg1\" \"arg2\" \"arg3\""
      "NUM_CMNDS 1"
    )

  MESSAGE("***\n*** Add two basic commands with 1 and two arguments\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_cmnd_2_args_1_2
    TEST_0 CMND echo ARGS "Cats and Dogs"
    TEST_1 CMND ls ARGS Cats
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    2
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"echo\" \"Cats and Dogs\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_1
    "\"ls\" \"Cats\""
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_cmnd_2_args_1_2.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"echo\" \"Cats and Dogs\""
      "TEST_1_CMND \"ls\" \"Cats\""
      "NUM_CMNDS 2"
    )

  MESSAGE("***\n*** Add a single basic command matching HOST\n***")
  SET(${PROJECT_NAME}_HOSTNAME MyHost)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_host
    TEST_0 CMND ${CMNDN}
    HOST MyHost
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMNDN}\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_PackageAddAdvancedTest_basic_host.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"ls\""
      "NUM_CMNDS 1"
    )

  MESSAGE("***\n*** Add a single basic command not matching HOST\n***")
  SET(${PROJECT_NAME}_HOSTNAME MyHost)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_basic_host
    TEST_0 CMND ${CMNDN}
    HOST NotMyHost
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  # ToDo: Add 6 more tests testing XHOST, HOSTTYPE, and XHOSTTYPE

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_ADVANCED_TEST_CATEGORIES)

  MESSAGE("\n***")
  MESSAGE("*** Testing TRIBITS_ADD_ADVANCED_TEST( ... CATEGORIES ... )")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageA)

  SET(${PROJECT_NAME}_TEST_CATEGORIES "")

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN}.exe)
  
  MESSAGE("\n*** Test empty CATEGORIES matching the BASIC category\n")

  MESSAGE("Test empty CATEGORIES matching CONTINUOUS category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Nightly)
  SET(${PROJECT_NAME}_TEST_CATEGORIES CONTINUOUS)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("Test empty CATEGORIES matching NIGHTLY category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Nightly)
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("Test empty CATEGORIES *not* matching PERFORMANCE category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES PERFORMANCE)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )
  
  MESSAGE("\n*** Test CATEGORIES BASIC\n")

  MESSAGE("Test CATEGORIES BASIC matching BASIC category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES BASIC )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("Test CATEGORIES BASIC matching CONTINUOUS category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES CONTINUOUS)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES BASIC )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("Test CATEGORIES BASIC matching NIGHTLY category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES BASIC )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("\n*** Test CATEGORIES CONTINUOUS\n")

  MESSAGE("Test CATEGORIES CONTINUOUS *not* matching BASIC category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES CONTINUOUS )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test CATEGORIES CONTINUOUS matching CONTINUOUS category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES CONTINUOUS)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES CONTINUOUS )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("Test CATEGORIES CONTINUOUS matching NIGHTLY category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES CONTINUOUS )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("\n*** Test CATEGORIES NIGHTLY\n")

  MESSAGE("Test CATEGORIES NIGHTLY *not* matching BASIC category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES NIGHTLY )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test CATEGORIES NIGHTLY *not* matching CONTINUOUS category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES CONTINUOUS)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES NIGHTLY )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test CATEGORIES NIGHTLY matching NIGHTLY category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES NIGHTLY )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  MESSAGE("\n*** Test CATEGORIES PERFORMANCE\n")

  MESSAGE("Test CATEGORIES PERFORMANCE *not* matching BASIC category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES BASIC)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES PERFORMANCE )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test CATEGORIES PERFORMANCE *not* matching CONTINUOUS category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES CONTINUOUS)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES PERFORMANCE )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test CATEGORIES PERFORMANCE *not* matching NIGHTLY category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Empty_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES PERFORMANCE )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("Test PERFORMANCE category matching PERFORMANCE category set by client")
  SET(TEST_NAME PackageAddAdvancedTestCategory_Performance_Performance)
  SET(${PROJECT_NAME}_TEST_CATEGORIES PERFORMANCE)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} CATEGORIES PERFORMANCE )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 1"
    )

  # NOTE: The above tests ensure that the CATEGORIES argument is accepted and
  # processed correctly.  The unit tests in
  # UNIT_TEST_PACKAGE_ADD_TEST_CATEGORIES() test the behavior of the logic for
  # selecting tests based on CATEGORIES.

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_ADVANCED_TEST_COMM)

  MESSAGE("\n***")
  MESSAGE("*** Testing TRIBITS_ADD_ADVANCED_TEST( ... COMM ... )")
  MESSAGE("***\n")

  # Needed by TRIBITS_ADD_ADVANCED_TEST(...)
  SET(PACKAGE_NAME PackageA)

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN})
  SET(MPI_EXEC_MAX_NUMPROCS 5)
  SET(MPI_EXEC mpiexec)
  SET(MPI_EXEC_PRE_NUMPROCS_FLAGS --pre-num-procs-flags)
  SET(MPI_EXEC_NUMPROCS_FLAG --num-procs)
  SET(MPI_EXEC_POST_NUMPROCS_FLAGS --post-num-procs-flags)
  SET(CMNDN ls)

  SET(PACKAGE_ADD_ADVANCED_TEST_SKIP_SCRIPT TRUE)

  # Default serial mode

  MESSAGE("***\n*** Add a test with no COMM argument\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( DummyTest
    TEST_0 CMND ${CMNDN}
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMNDN}\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )

  MESSAGE("***\n*** Add a serial test with 'COMM serial'\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( DummyTest
    TEST_0 CMND ${CMNDN}
    COMM serial
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${CMNDN}\""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )

  # Doing MPI mode
  SET(TPL_ENABLE_MPI ON)

  MESSAGE("***\n*** Add serial-only test with 'COMM mpi' (will not add the test)\n***")
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( DummyTest
    TEST_0 CMND ${CMNDN}
    COMM serial
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    ""
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    ""
    )

  MESSAGE("***\n*** Add an advanced test for MPI with no COMM input but with two args\n***")
  GLOBAL_SET(PACKAGE_ADD_TEST_ADD_TEST_INPUT)
  TRIBITS_ADD_ADVANCED_TEST( PackageAddAdvancedTest_mpi_exec_1_args_2
    TEST_0 EXEC ${EXEN} ARGS arg1 arg2 )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_NUM_CMNDS
    1
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_ADVANCED_TEST_CMND_ARRAY_0
    "\"${MPI_EXEC}\" \"${MPI_EXEC_PRE_NUMPROCS_FLAGS}\" \"${MPI_EXEC_NUMPROCS_FLAG}\" \"${MPI_EXEC_MAX_NUMPROCS}\" \"${MPI_EXEC_POST_NUMPROCS_FLAGS}\" \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}.exe\" \"arg1\" \"arg2\""
    )

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_ADVANCED_TEST_DIRECTROY)

  MESSAGE("\n***")
  MESSAGE("*** Testing TRIBITS_ADD_ADVANCED_TEST( ... DIRECTORY ... )")
  MESSAGE("***\n")


  # Needed by TRIBITS_ADD_TEST(...)
  SET(PACKAGE_NAME PackageA)

  SET(PACKAGE_ADD_ADVANCED_TEST_SKIP_SCRIPT FALSE)

  # Used locally
  SET(EXEN SomeExec)
  SET(PACKEXEN ${PACKAGE_NAME}_${EXEN}.exe)
  
  MESSAGE("\n*** Two tests with no DIRECTORY argument \n")
  SET(TEST_NAME PackageAddAdvancedTestDirectory_None)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} TEST_1 EXEC ${EXEN} )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "TEST_1_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 2"
    )
  
  MESSAGE("\n*** Two tests, first test with DIRECTORY argument \n")
  SET(TEST_NAME PackageAddAdvancedTestDirectory_None)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} DIRECTORY ../dir1 TEST_1 EXEC ${EXEN} )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/../dir1/${PACKEXEN}\""
      "TEST_1_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "NUM_CMNDS 2"
    )
  
  MESSAGE("\n*** Two tests, second test with DIRECTORY argument \n")
  SET(TEST_NAME PackageAddAdvancedTestDirectory_None)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} TEST_1 EXEC ${EXEN} DIRECTORY ../dir2 )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/${PACKEXEN}\""
      "TEST_1_CMND \"${CMAKE_CURRENT_BINARY_DIR}/../dir2/${PACKEXEN}\""
      "NUM_CMNDS 2"
    )
  
  MESSAGE("\n*** Two tests, both tests with DIRECTORY argument \n")
  SET(TEST_NAME PackageAddAdvancedTestDirectory_None)
  TRIBITS_ADD_ADVANCED_TEST_UNITTEST_RESET()
  TRIBITS_ADD_ADVANCED_TEST( ${TEST_NAME} TEST_0 EXEC ${EXEN} DIRECTORY ../dir1 TEST_1 EXEC ${EXEN} DIRECTORY ../dir2 )
  UNITTEST_FILE_REGEX(
    "${CMAKE_CURRENT_BINARY_DIR}/${PACKAGE_NAME}_${TEST_NAME}.cmake"
    REGEX_STRINGS
      "TEST_0_CMND \"${CMAKE_CURRENT_BINARY_DIR}/../dir1/${PACKEXEN}\""
      "TEST_1_CMND \"${CMAKE_CURRENT_BINARY_DIR}/../dir2/${PACKEXEN}\""
      "NUM_CMNDS 2"
    )

ENDFUNCTION()


FUNCTION(UNITEST_PACKAGE_ADD_EXECUTABLE_AND_TEST)

  SET(PACKAGE_ADD_EXECUTABLE_AND_TEST_TEST_MODE ON)

  MESSAGE("\n***")
  MESSAGE("*** Test passing basic arguments to TRIBITS_ADD_EXECUTABLE_AND_TEST( ... )")
  MESSAGE("***\n")

  TRIBITS_ADD_EXECUTABLE_AND_TEST( 
    execName
    SOURCES src1 src2
    NAME testName
    NAME_POSTFIX testNamePostfix
    CATEGORIES category1 category2
    HOST host1 host2
    XHOST host1 host2
    HOSTTYPE hosttype1 hosttype2
    XHOSTTYPE hosttype1 hosttype2
    NOEXEPREFIX
    NOEXESUFFIX
    DIRECTORY dir
    DEPLIBS lib1 lib2
    COMM serial mpi
    NUM_MPI_PROCS numProcs
    LINKER_LANGUAGE C
    ADD_DIR_TO_NAME
    DEFINES -DSOMEDEFINE
    KEYWORDS keyword1 keyword2
    STANDARD_PASS_OUTPUT
    PASS_REGULAR_EXPRESSION "regex1;regex2"
    FAIL_REGULAR_EXPRESSION "regex1;regex2"
    ENVIRONMENT env1=envval1 env2=envval2
    WILL_FAIL
    TIMEOUT
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_EXECUTABLE_CAPTURE_ARGS
    "execName;COMM;serial;mpi;CATEGORIES;category1;category2;HOST;host1;host2;XHOST;host1;host2;HOSTTYPE;hosttype1;hosttype2;XHOSTTYPE;hosttype1;hosttype2;NOEXEPREFIX;NOEXESUFFIX;SOURCES;src1;src2;DEPLIBS;lib1;lib2;DIRECTORY;dir;ADD_DIR_TO_NAME;LINKER_LANGUAGE;C;DEFINES;-DSOMEDEFINE"
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_CAPTURE_ARGS
    "execName;COMM;serial;mpi;CATEGORIES;category1;category2;HOST;host1;host2;XHOST;host1;host2;HOSTTYPE;hosttype1;hosttype2;XHOSTTYPE;hosttype1;hosttype2;NOEXEPREFIX;NOEXESUFFIX;NAME;testName;NAME_POSTFIX;testNamePostfix;DIRECTORY;dir;KEYWORDS;keyword1;keyword2;NUM_MPI_PROCS;numProcs;PASS_REGULAR_EXPRESSION;regex1;regex2;FAIL_REGULAR_EXPRESSION;regex1;regex2;ENVIRONMENT;env1=envval1;env2=envval2;STANDARD_PASS_OUTPUT;WILL_FAIL;TIMEOUT;ADD_DIR_TO_NAME"
    )

  MESSAGE("\n***")
  MESSAGE("*** Test passing in XHOST_TEST and XHOSTTYPE_TEST into TRIBITS_ADD_EXECUTABLE_AND_TEST(...)")
  MESSAGE("***\n")

  TRIBITS_ADD_EXECUTABLE_AND_TEST( 
    execName
    SOURCES src1 src2
    XHOST_TEST host1 host2
    XHOSTTYPE_TEST hosttype1 hosttype2
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_EXECUTABLE_CAPTURE_ARGS
    "execName;SOURCES;src1;src2"
    )
  UNITTEST_COMPARE_CONST(
    PACKAGE_ADD_TEST_CAPTURE_ARGS
    "execName;XHOST;host1;host2;XHOSTTYPE;hosttype1;hosttype2"
    )

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ETI_TYPE_EXPANSION)

  MESSAGE("*** Test passing invalid arguments to TRIBITS_ETI_TYPE_EXPANSION( ... )\n")

  UNSET(expansion)
  TRIBITS_ETI_TYPE_EXPANSION(expansion "badformat")
  UNITTEST_COMPARE_CONST(
    expansion
    "TRIBITS_ETI_BAD_ARGUMENTS"
    )

  MESSAGE("*** Test passing valid arguments to TRIBITS_ETI_TYPE_EXPANSION( ... )\n")

  # test rank-one
  UNSET(expansion)
  TRIBITS_ETI_TYPE_EXPANSION(expansion "f1=ta|tb")
  UNITTEST_COMPARE_CONST(
    expansion
    "f1={ta};f1={tb}"
    )

  # test accumulation into ${expansion}
  TRIBITS_ETI_TYPE_EXPANSION(expansion "f2=tc|td|te")
  UNITTEST_COMPARE_CONST(
    expansion
    "f1={ta};f1={tb};f2={tc};f2={td};f2={te}"
    )

  # test rank-two
  UNSET(expansion)
  TRIBITS_ETI_TYPE_EXPANSION(expansion "f1=ta|tb" "f2=tc")
  UNITTEST_COMPARE_CONST(
    expansion
    "f1={ta} f2={tc};f1={tb} f2={tc}"
    )

  # test rank-three
  UNSET(expansion)
  TRIBITS_ETI_TYPE_EXPANSION(expansion "f1=ta|tb" "f2=tc" "f3=td|te")
  UNITTEST_COMPARE_CONST(
    expansion
    "f1={ta} f2={tc} f3={td};f1={ta} f2={tc} f3={te};f1={tb} f2={tc} f3={td};f1={tb} f2={tc} f3={te}"
    )

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ETI_CHECK_EXCLUSION)

  MESSAGE("*** Test passing valid arguments to TRIBITS_ETI_CHECK_EXCLUSION( ... )\n")

  message("empty exclusion list...")
  TRIBITS_ETI_CHECK_EXCLUSION("" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    OFF
  )

  message("inst not excluded (no match)...")
  TRIBITS_ETI_CHECK_EXCLUSION("td|te|tf" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    OFF
  )

  message("matches only on present types...")
  TRIBITS_ETI_CHECK_EXCLUSION("ta|ta|tb" "ta|TYPE-MISSING|tb" result)
  UNITTEST_COMPARE_CONST(
    result
    ON
  )

  message("no match: exclusion has the wrong rank (not an error)...")
  TRIBITS_ETI_CHECK_EXCLUSION("ta|ta" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    OFF
  )

  message("inst not excluded (partial match)...")
  TRIBITS_ETI_CHECK_EXCLUSION("ta|tb|ta;tb|tb|tc;ta|ta|tc" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    OFF
  )

  message("inst exluded (full explicit)...")
  TRIBITS_ETI_CHECK_EXCLUSION("abcdf;ta|tb|tc" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    ON
  )

  message("inst exluded (full regex)...")
  TRIBITS_ETI_CHECK_EXCLUSION("abcdf;.*|.*|.*" "ta|tb|tc" result)
  UNITTEST_COMPARE_CONST(
    result
    ON
  )

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ETI_INDEX_MACRO_FIELDS)

  MESSAGE("*** Test passing valid arguments to TRIBITS_ETI_INDEX_MACRO_FIELDS( ... )\n")

  # check simple
  TRIBITS_ETI_INDEX_MACRO_FIELDS("F1;F2;F3" "F3" var)
  UNITTEST_COMPARE_CONST(
    var
    "2"
    )

  # check complex
  TRIBITS_ETI_INDEX_MACRO_FIELDS("F1;F2;F3" "F3;F2;F1" var)
  UNITTEST_COMPARE_CONST(
    var
    "2;1;0"
    )

  # check complex with spaces
  TRIBITS_ETI_INDEX_MACRO_FIELDS("F1;F2;F3" " F2 ;   F2 ; F2 " var)
  UNITTEST_COMPARE_CONST(
    var
    "1;1;1"
    )

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ADD_ETI_INSTANTIATIONS_INITIAL)

  MESSAGE("*** Testing TRIBITS_ADD_ETI_INSTANTIATIONS... )\n")

  SET(package ${PROJECT_NAME}Framework)
  GLOBAL_NULL_SET(${package}_ETI_LIBRARYSET)
  TRIBITS_ADD_ETI_INSTANTIATIONS(${package} "someinst")
  UNITTEST_COMPARE_CONST(
    ${package}_ETI_LIBRARYSET
    "someinst"
    )

ENDFUNCTION()

FUNCTION(UNITTEST_TRIBITS_ADD_ETI_INSTANTIATIONS_CUMULATIVE)

  SET(package ${PROJECT_NAME}Framework)
  TRIBITS_ADD_ETI_INSTANTIATIONS(${package} "anotherinst")
  UNITTEST_COMPARE_CONST(
    ${package}_ETI_LIBRARYSET
    "someinst;anotherinst"
    )

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ETI_EXPLODE)

  MESSAGE("*** Test passing valid arguments to TRIBITS_ETI_EXPLODE( ... )\n")

  # no fields -> no results
  SET(FIELDS "")
  TRIBITS_ETI_EXPLODE("${FIELDS}" "F1=type1 F2=type2 F3=type3" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    ""
    )

  # order doesn't matter; also, results should be bracketed
  SET(FIELDS F FF G)
  TRIBITS_ETI_EXPLODE("${FIELDS}" "F=type1 FF=type2 G={type3}" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "type1|type2|type3"
    )
  TRIBITS_ETI_EXPLODE("${FIELDS}" "FF=type2 F={type1} G=type3" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "type1|type2|type3"
    )
  TRIBITS_ETI_EXPLODE("${FIELDS}" "G=type3 FF=type2 F={type1}" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "type1|type2|type3"
    )
  
  # empty for missing fields

  # missing field handled properly, extra fields ignored
  SET(FIELDS F FF G)
  TRIBITS_ETI_EXPLODE("${FIELDS}" "F=type1 G=type3 H=type4" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "type1|TYPE-MISSING|type3"
    )

  # bad bracketing doesn't work
  TRIBITS_ETI_EXPLODE("F" "F=typea}" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "TRIBITS_ETI_BAD_PARSE"
    )
  TRIBITS_ETI_EXPLODE("F" "F={typea" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "TRIBITS_ETI_BAD_PARSE"
    )
  TRIBITS_ETI_EXPLODE("F" "F={typea}}" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "TRIBITS_ETI_BAD_PARSE"
    )
  TRIBITS_ETI_EXPLODE("F" "F={{typea}" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "TRIBITS_ETI_BAD_PARSE"
    )
  TRIBITS_ETI_EXPLODE("F" "F=typeaG=typeb" parsed)
  UNITTEST_COMPARE_CONST(
    parsed
    "TRIBITS_ETI_BAD_PARSE"
    )

ENDFUNCTION()

FUNCTION(UNITTEST_TRIBITS_ETI_MANGLE_SYMBOL)

  MESSAGE("*** Testing ETI Mangling ***")

  # this one is ugly...
  TRIBITS_ETI_MANGLE_SYMBOL(mangled "std::pair< std::complex< double > , std::complex< float > >")
  UNITTEST_COMPARE_CONST(
    mangled
    "std_pair2std_complex1double1_std_complex0float02")

  # test that POD isn't mangled, and that the method accumulates into the typedef list
  SET(defs_orig "do not delete")
  SET(defs "${defs_orig}")
  SET(symbol "double")
  SET(mangling_list "")
  TRIBITS_ETI_MANGLE_SYMBOL_AUGMENT_MACRO(defs symbol mangling_list)
  UNITTEST_COMPARE_CONST(
    symbol
    "double")
  UNITTEST_COMPARE_CONST(
    defs
    "${defs_orig}")
  UNITTEST_COMPARE_CONST(
    mangling_list
    "")

  # this is more like what we expect
  SET(defs "")
  SET(mangling_list "")
  #
  SET(symbol "std::complex<float>")
  TRIBITS_ETI_MANGLE_SYMBOL_AUGMENT_MACRO(defs symbol mangling_list)
  UNITTEST_COMPARE_CONST(
    symbol
    "std_complex0float0")
  #
  SET(symbol "std::pair<float,float>")
  TRIBITS_ETI_MANGLE_SYMBOL_AUGMENT_MACRO(defs symbol mangling_list)
  UNITTEST_COMPARE_CONST(
    symbol
    "std_pair0float_float0")
  #
  UNITTEST_COMPARE_CONST(
    mangling_list
    "std_complex0float0;std_pair0float_float0")
  UNITTEST_COMPARE_CONST(
    defs
    "typedef std::complex<float> std_complex0float0;typedef std::pair<float,float> std_pair0float_float0")

ENDFUNCTION()


FUNCTION(UNITTEST_TRIBITS_ETI_GENERATE_MACROS)

  MESSAGE("*** Test TRIBITS_ETI_GENERATE_MACROS( ... )\n")

  TRIBITS_ETI_TYPE_EXPANSION(
    etiset 
    "F1=Teuchos::ArrayRCP<Teuchos::ArrayRCP<double> > | double" 
    "F2=int | long"
    "F3=float"
  )
  TRIBITS_ETI_TYPE_EXPANSION(
    exclset
    "F1=.*"
    "F2=long"
    "F3=.*"
  )
  TRIBITS_ETI_GENERATE_MACROS(
    "F1|F2|F3" 
    "${etiset}" 
    "${exclset}"
    mangling_list     typedef_list
    "F1(F1)"          macro_f1_var
    "F312(F3,F1,F2)"  macro_f312_var
  )
  UNITTEST_COMPARE_CONST(
    macro_f1_var
"#define F1(INSTMACRO)\\
\tINSTMACRO( Teuchos_ArrayRCP1Teuchos_ArrayRCP0double01 )\\
\tINSTMACRO( double )
"
    )
  UNITTEST_COMPARE_CONST(
    macro_f312_var
"#define F312(INSTMACRO)\\
\tINSTMACRO( float , Teuchos_ArrayRCP1Teuchos_ArrayRCP0double01 , int )\\
\tINSTMACRO( float , double , int )
"
    )
  UNITTEST_COMPARE_CONST(
    typedef_list 
    "typedef Teuchos::ArrayRCP<Teuchos::ArrayRCP<double> > Teuchos_ArrayRCP1Teuchos_ArrayRCP0double01")
  UNITTEST_COMPARE_CONST(
    mangling_list 
    "Teuchos_ArrayRCP1Teuchos_ArrayRCP0double01")

  SET(mangling_list "")
  SET(typedef_list  "")
  TRIBITS_ETI_GENERATE_MACROS(
    "F1|F2"
    "F1=a F2=b;F2=c;G1=d G2=e;G3=f"
    ""
    mangling_list     typedef_list
    "F2(F2)"          macro_f2_var
    "F12(F1,F2)"      macro_f12_var
  )
  UNITTEST_COMPARE_CONST( typedef_list "")
  UNITTEST_COMPARE_CONST( mangling_list "")
  UNITTEST_COMPARE_CONST(
    macro_f2_var
"#define F2(INSTMACRO)\\
\tINSTMACRO( b )\\
\tINSTMACRO( c )
")
  UNITTEST_COMPARE_CONST(
    macro_f12_var
"#define F12(INSTMACRO)\\
\tINSTMACRO( a , b )
")

ENDFUNCTION()

#
# Execute the unit tests
#

# Set up some global environment stuff
SET(${PROJECT_NAME}_HOSTNAME testhost.nowhere.com)
SET(CMAKE_HOST_SYSTEM_NAME UnspecifiedHostSystemName)

# Assume that all unit tests will pass by default
GLOBAL_SET(UNITTEST_OVERALL_PASS TRUE)
GLOBAL_SET(UNITTEST_OVERALL_NUMPASSED 0)
GLOBAL_SET(UNITTEST_OVERALL_NUMRUN 0)

# Set up the TRIBITS_ADD_TEST(...) function to only capture the output and not
# actually add the test.
SET( PACKAGE_ADD_TEST_ADD_TEST_CAPTURE TRUE )
SET( PACKAGE_ADD_TEST_ADD_TEST_SKIP TRUE )

# Don't let PACKAGE_ADD_ADVANCED_TEST(...) actually call ADD_TEST(...).
SET( PACKAGE_ADD_ADVANCED_TEST_SKIP_SCRIPT_ADD_TEST TRUE )

UNIT_TEST_APPEND_STRING_VAR()
UNITEST_PACKAGE_ARCH_MISC()

# Set the default test categories
SET(${PROJECT_NAME}_TEST_CATEGORIES NIGHTLY)

MESSAGE("\n***")
MESSAGE("*** Testing PACKAGE_ADD_TEST(...)")
MESSAGE("***\n")

UNITEST_PACKAGE_ADD_TEST_BASIC()
UNITEST_PACKAGE_ADD_TEST_CATEGORIES()
UNITEST_PACKAGE_ADD_TEST_COMM()


MESSAGE("\n***")
MESSAGE("*** Testing PACKAGE_ADD_ADVANCED_TEST(...)")
MESSAGE("***\n")

UNITEST_PACKAGE_ADD_ADVANCED_TEST_BASIC()
UNITEST_PACKAGE_ADD_ADVANCED_TEST_CATEGORIES()
UNITEST_PACKAGE_ADD_ADVANCED_TEST_COMM()
UNITEST_PACKAGE_ADD_ADVANCED_TEST_DIRECTROY()

UNITEST_PACKAGE_ADD_TEST_DISABLE()

MESSAGE("\n***")
MESSAGE("*** Testing PACKAGE_ADD_EXECUTABLE_AND_TEST(...)")
MESSAGE("***\n")

UNITEST_PACKAGE_ADD_EXECUTABLE_AND_TEST()

MESSAGE("\n***")
MESSAGE("*** Testing Explicit Template Instantiation functionality")
MESSAGE("***\n")

UNITTEST_TRIBITS_ETI_EXPLODE()
UNITTEST_TRIBITS_ETI_TYPE_EXPANSION()
UNITTEST_TRIBITS_ETI_CHECK_EXCLUSION()
UNITTEST_TRIBITS_ETI_INDEX_MACRO_FIELDS()
UNITTEST_TRIBITS_ADD_ETI_INSTANTIATIONS_INITIAL()
UNITTEST_TRIBITS_ADD_ETI_INSTANTIATIONS_CUMULATIVE()
UNITTEST_TRIBITS_ETI_MANGLE_SYMBOL()
UNITTEST_TRIBITS_ETI_GENERATE_MACROS()

MESSAGE("\n***")
MESSAGE("*** Determine final result of all unit tests")
MESSAGE("***\n")

# Pass in the number of expected tests that must pass!
UNITTEST_FINAL_RESULT(195)
