//@HEADER
// ************************************************************************
//
//          Kokkos: Node API and Parallel Node Kernels
//              Copyright (2008) Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Michael A. Heroux (maherou@sandia.gov)
//
// ************************************************************************
//@HEADER

#ifndef __Kokkos_Raw_SparseMatVec_def_hpp
#define __Kokkos_Raw_SparseMatVec_def_hpp

/// \file Kokkos_Raw_SparseMatVec_def.hpp
/// \brief Definitions of "raw" sequential and OpenMP-parallel
///   sparse matrix-(multi)vector multiply routines.
/// \warning This code was generated by the SparseMatVec.py script.
///   If you edit this header by hand, your edits will disappear the
///   next time you run the generator script.

namespace Kokkos {
namespace Raw {

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += A_ij * tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] -= A_ij * tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += alpha * A_ij * tmp;
        }
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor4Unrolled (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] += A_ij * tmp[0];
          Y_i[colStrideY] += A_ij * tmp[1];
          Y_i[2*colStrideY] += A_ij * tmp[2];
          Y_i[3*colStrideY] += A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += A_ij * tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] -= A_ij * tmp[0];
          Y_i[colStrideY] -= A_ij * tmp[1];
          Y_i[2*colStrideY] -= A_ij * tmp[2];
          Y_i[3*colStrideY] -= A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] -= A_ij * tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] += alpha * A_ij * tmp[0];
          Y_i[colStrideY] += alpha * A_ij * tmp[1];
          Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
          Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += alpha * A_ij * tmp;
        }
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        Y[i] += A_ij * tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        Y[i] -= A_ij * tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        Y[i] += alpha * A_ij * tmp;
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
        Y_i[2*colStrideY] += A_ij * tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
        Y_i[2*colStrideY] -= A_ij * tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
        Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForfor4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
        Y_i[2*colStrideY] += A_ij * tmp[2];
        Y_i[3*colStrideY] += A_ij * tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
        Y_i[2*colStrideY] -= A_ij * tmp[2];
        Y_i[3*colStrideY] -= A_ij * tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
        Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
        Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhile1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += A_ij * tmp;
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] -= A_ij * tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += alpha * A_ij * tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhile2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhile3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhile4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
      Y_i[3*colStrideY] += A_ij * tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
      Y_i[3*colStrideY] -= A_ij * tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForif1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += A_ij * tmp;
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] -= A_ij * tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += alpha * A_ij * tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForif2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForif3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForif4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
      Y_i[3*colStrideY] += A_ij * tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
      Y_i[3*colStrideY] -= A_ij * tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = val[k];
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor4Unrolled (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor4UnrolledOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = val[k];
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      const RangeScalar zero = STS::zero ();
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = zero;

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      const RangeScalar zero = STS::zero ();
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = zero;

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      size_t k = 0;
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        const size_t k_end = ptr[i+1];
        for ( ; k < k_end; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor1VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor2VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor3VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForfor4VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = val[k];
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhile1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhile2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhile3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhile4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForif1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForif2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForif3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForif4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = val[k];
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += A_ij * tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] -= A_ij * tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      for (Ordinal c = 0; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += alpha * A_ij * tmp;
        }
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj4Unrolled (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] += A_ij * tmp[0];
          Y_i[colStrideY] += A_ij * tmp[1];
          Y_i[2*colStrideY] += A_ij * tmp[2];
          Y_i[3*colStrideY] += A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += A_ij * tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] -= A_ij * tmp[0];
          Y_i[colStrideY] -= A_ij * tmp[1];
          Y_i[2*colStrideY] -= A_ij * tmp[2];
          Y_i[3*colStrideY] -= A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] -= A_ij * tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      // Extra +1 in loop bound ensures first 4 iterations get
      // strip-mined, but requires that Ordinal be a signed type.
      Ordinal c = 0;
      for ( ; c < numVecs - 3; c += 4) {
        const DomainScalar* const X_j = &X[j + c*colStrideX];
        const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          Y_i[0] += alpha * A_ij * tmp[0];
          Y_i[colStrideY] += alpha * A_ij * tmp[1];
          Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
          Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
        }
      }
      // Mop up left-over iterations over multivector columns.
      for ( ; c < numVecs; ++c) {
        const DomainScalar tmp = X[j + c*colStrideX];

        for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal i = ind[k];
          Y[i + c*colStrideY] += alpha * A_ij * tmp;
        }
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        Y[i] += A_ij * tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        Y[i] -= A_ij * tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar tmp = X[j];

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        Y[i] += alpha * A_ij * tmp;
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[2] = {X_j[0], X_j[colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
        Y_i[2*colStrideY] += A_ij * tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
        Y_i[2*colStrideY] -= A_ij * tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[3] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
        Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForforConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += A_ij * tmp[0];
        Y_i[colStrideY] += A_ij * tmp[1];
        Y_i[2*colStrideY] += A_ij * tmp[2];
        Y_i[3*colStrideY] += A_ij * tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] -= A_ij * tmp[0];
        Y_i[colStrideY] -= A_ij * tmp[1];
        Y_i[2*colStrideY] -= A_ij * tmp[2];
        Y_i[3*colStrideY] -= A_ij * tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    for (Ordinal j = 0; j < numCols; ++j) {
      const DomainScalar* const X_j = &X[j];
      const DomainScalar tmp[4] = {X_j[0], X_j[colStrideX], X_j[2*colStrideX], X_j[3*colStrideX]};

      for (size_t k = ptr[j]; k < ptr[j+1]; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal i = ind[k];
        RangeScalar* const Y_i = &Y[i];
        Y_i[0] += alpha * A_ij * tmp[0];
        Y_i[colStrideY] += alpha * A_ij * tmp[1];
        Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
        Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
      }
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhileConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += A_ij * tmp;
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] -= A_ij * tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += alpha * A_ij * tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhileConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhileConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForwhileConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
      Y_i[3*colStrideY] += A_ij * tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
      Y_i[3*colStrideY] -= A_ij * tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      while (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForifConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += A_ij * tmp;
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] -= A_ij * tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp = Teuchos::ScalarTraits<DomainScalar>::zero();
    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        tmp = X[j + 0*colStrideX];
      }
      Y[i] += alpha * A_ij * tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForifConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[2];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForifConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[3];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCscColMajorForifConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Prescale: Y := beta * Y.
  if (beta == STS::zero()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        // Follow the Sparse BLAS convention for beta == 0.
        Y_j[i] = STS::zero();
      }
    }
  }
  else if (beta != STS::one()) {
    for (Ordinal j = 0; j < numVecs; ++j) {
      RangeScalar* const Y_j = &Y[j*colStrideY];
      for (Ordinal i = 0; i < numRows; ++i) {
        Y_j[i] = beta * Y_j[i];
      }
    }
  }
  // Outer for loop preface:
  if (alpha == STS::zero()) {
    return; // Our work is done!
  }
  const size_t nnz = ptr[numCols];
  if (alpha == STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += A_ij * tmp[0];
      Y_i[colStrideY] += A_ij * tmp[1];
      Y_i[2*colStrideY] += A_ij * tmp[2];
      Y_i[3*colStrideY] += A_ij * tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] -= A_ij * tmp[0];
      Y_i[colStrideY] -= A_ij * tmp[1];
      Y_i[2*colStrideY] -= A_ij * tmp[2];
      Y_i[3*colStrideY] -= A_ij * tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    // Invariant: Right before updating Y(i,:), tmp = X(j,:).
    // Initializing tmp here isn't necessary for correctness, but it
    // makes compilers stop complaining about uninitialized variables.
    DomainScalar tmp[4];
    tmp[0] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[1] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[2] = Teuchos::ScalarTraits<DomainScalar>::zero();
    tmp[3] = Teuchos::ScalarTraits<DomainScalar>::zero();

    Ordinal j = 0;
    for (size_t k = 0; k < nnz; ++k) {
      const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
      const Ordinal i = ind[k];
      // NOTE: "if" instead of "while" here is only valid
      // if the matrix contains no empty rows.
      if (k >= ptr[j+1]) {
        ++j;
        const DomainScalar* const X_j = &X[j];
        tmp[0] = X_j[0];
        tmp[1] = X_j[colStrideX];
        tmp[2] = X_j[2*colStrideX];
        tmp[3] = X_j[3*colStrideX];
      }
      RangeScalar* const Y_i = &Y[i];
      Y_i[0] += alpha * A_ij * tmp[0];
      Y_i[colStrideY] += alpha * A_ij * tmp[1];
      Y_i[2*colStrideY] += alpha * A_ij * tmp[2];
      Y_i[3*colStrideY] += alpha * A_ij * tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConjOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        for (Ordinal c = 0; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          // Copy temporary values into output vector.
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj4Unrolled (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj4UnrolledOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += A_ij * X_j[0];
            tmp[1] += A_ij * X_j[colStrideX];
            tmp[2] += A_ij * X_j[2*colStrideX];
            tmp[3] += A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] -= A_ij * X_j[0];
            tmp[1] -= A_ij * X_j[colStrideX];
            tmp[2] -= A_ij * X_j[2*colStrideX];
            tmp[3] -= A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp -= A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = -Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = STS::zero();

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        // Extra +1 in loop bound ensures first 4 iterations get
        // strip-mined, but requires that Ordinal be a signed type.
        Ordinal c = 0;
        for ( ; c < numVecs - 3; c += 4) {
          RangeScalar* const Y_i = &Y[i + c*colStrideY];
          RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            const DomainScalar* const X_j = &X[j + c*colStrideX];
            tmp[0] += alpha * A_ij * X_j[0];
            tmp[1] += alpha * A_ij * X_j[colStrideX];
            tmp[2] += alpha * A_ij * X_j[2*colStrideX];
            tmp[3] += alpha * A_ij * X_j[3*colStrideX];
          }
          // Copy temporary values into output vector.
          Y_i[0] = tmp[0];
          Y_i[colStrideY] = tmp[1];
          Y_i[2*colStrideY] = tmp[2];
          Y_i[3*colStrideY] = tmp[3];
        }
        // Mop up left-over iterations over multivector columns.
        for ( ; c < numVecs; ++c) {
          RangeScalar tmp = beta * Y[i + c*colStrideY];

          for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
            const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
            const Ordinal j = ind[k];
            tmp += alpha * A_ij * X[j + c*colStrideX];
          }
          Y[i + c*colStrideY] = tmp;
        }
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj1VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp -= A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar tmp = -Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar tmp = STS::zero();

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar tmp = Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar tmp = beta * Y[i];

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          tmp += alpha * A_ij * X[j];
        }
        // Copy temporary values into output vector.
        Y[i] = tmp;
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj2VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {-Y_i[0], -Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {Y_i[0], Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[2] = {beta * Y_i[0], beta * Y_i[colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj3VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[3] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
}

#ifdef HAVE_KOKKOSCLASSIC_OPENMP
template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForforConj4VecOmp (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // With CSR for alpha == 0, scale Y by beta and return.
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        #pragma omp parallel for
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += A_ij * X_j[0];
          tmp[1] += A_ij * X_j[colStrideX];
          tmp[2] += A_ij * X_j[2*colStrideX];
          tmp[3] += A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] -= A_ij * X_j[0];
          tmp[1] -= A_ij * X_j[colStrideX];
          tmp[2] -= A_ij * X_j[2*colStrideX];
          tmp[3] -= A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to -Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {-Y_i[0], -Y_i[colStrideY], -Y_i[2*colStrideY], -Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::zero()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to 0.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {STS::zero(), STS::zero(), STS::zero(), STS::zero()};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else if (beta == STS::one()) {
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:).
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {Y_i[0], Y_i[colStrideY], Y_i[2*colStrideY], Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
    else { // beta != -1 && beta != 0 && beta != 1
      #pragma omp parallel for
      for (Ordinal i = 0; i < numRows; ++i) {
        // Initialize temporary values to Y(i,:) * beta.
        RangeScalar* const Y_i = &Y[i];
        RangeScalar tmp[4] = {beta * Y_i[0], beta * Y_i[colStrideY], beta * Y_i[2*colStrideY], beta * Y_i[3*colStrideY]};

        for (size_t k = ptr[i]; k < ptr[i+1]; ++k) {
          const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
          const Ordinal j = ind[k];
          const DomainScalar* const X_j = &X[j];
          tmp[0] += alpha * A_ij * X_j[0];
          tmp[1] += alpha * A_ij * X_j[colStrideX];
          tmp[2] += alpha * A_ij * X_j[2*colStrideX];
          tmp[3] += alpha * A_ij * X_j[3*colStrideX];
        }
        // Copy temporary values into output vector.
        Y_i[0] = tmp[0];
        Y_i[colStrideY] = tmp[1];
        Y_i[2*colStrideY] = tmp[2];
        Y_i[3*colStrideY] = tmp[3];
      }
    }
  }
}
#endif // HAVE_KOKKOSCLASSIC_OPENMP

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhileConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhileConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhileConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForwhileConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        while (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForifConj1Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp -= A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp = -Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp = STS::zero();
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else if (beta == STS::one()) {
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp = Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp = Teuchos::ScalarTraits<RangeScalar>::zero();
      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp;
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp = beta * Y[i + 0*colStrideY];
        }
        tmp += alpha * A_ij * X[j];
      }
      Y_i[0] = tmp;
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForifConj2Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[2];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForifConj3Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[3];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
    }
  }
}

template<class Ordinal,
         class MatrixScalar,
         class DomainScalar,
         class RangeScalar>
void
matVecCsrColMajorForifConj4Vec (
  const Ordinal numRows,
  const Ordinal numCols,
  const Ordinal numVecs,
  const RangeScalar& beta,
  RangeScalar Y[],
  const Ordinal colStrideY,
  const RangeScalar& alpha,
  const size_t  ptr[],
  const Ordinal ind[],
  const MatrixScalar val[],
  const DomainScalar X[],
  const Ordinal colStrideX)
{
  typedef Teuchos::ScalarTraits<RangeScalar> STS;

  // Algorithm variants 'for-while' and 'for-if' need to set
  // Y(0,:) = 0, but only for the special case of CSR.
  if (beta != STS::zero()) {
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = beta * Y[c*colStrideY];
    }
  }
  else {
    // Follow the Sparse BLAS convention for beta == 0.
    for (Ordinal c = 0; c < numVecs; ++c) {
      Y[c*colStrideY] = STS::zero();
    }
  }
  if (alpha == STS::zero()) {
    // Prescale: Y := beta * Y.
    if (beta == STS::zero()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          // Follow the Sparse BLAS convention for beta == 0.
          Y_j[i] = STS::zero();
        }
      }
    }
    else if (beta != STS::one()) {
      for (Ordinal j = 0; j < numVecs; ++j) {
        RangeScalar* const Y_j = &Y[j*colStrideY];
        for (Ordinal i = 0; i < numRows; ++i) {
          Y_j[i] = beta * Y_j[i];
        }
      }
    }
    return; // Our work is done!
  }
  const size_t nnz = ptr[numRows];
  if (alpha == STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += A_ij * X_j[0];
        tmp[1] += A_ij * X_j[colStrideX];
        tmp[2] += A_ij * X_j[2*colStrideX];
        tmp[3] += A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else if (alpha == -STS::one()) {
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] -= A_ij * X_j[0];
        tmp[1] -= A_ij * X_j[colStrideX];
        tmp[2] -= A_ij * X_j[2*colStrideX];
        tmp[3] -= A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
  else { // alpha != 1 && alpha != -1
    if (beta == -STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to -Y(i,:).
          Y_i = &Y[i];
          tmp[0] = -Y_i[0];
          tmp[1] = -Y_i[colStrideY];
          tmp[2] = -Y_i[2*colStrideY];
          tmp[3] = -Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::zero()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; set Y(i,:) to 0.
          Y_i = &Y[i];
          tmp[0] = STS::zero();
          tmp[1] = STS::zero();
          tmp[2] = STS::zero();
          tmp[3] = STS::zero();
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else if (beta == STS::one()) {
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We don't have to set Y(i,:) here, since beta == 1.
          Y_i = &Y[i];
          tmp[0] = Y_i[0];
          tmp[1] = Y_i[colStrideY];
          tmp[2] = Y_i[2*colStrideY];
          tmp[3] = Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
    else { // beta != -1 && beta != 0 && beta != 1
      RangeScalar tmp[4];
      tmp[0] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[1] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[2] = Teuchos::ScalarTraits<RangeScalar>::zero();
      tmp[3] = Teuchos::ScalarTraits<RangeScalar>::zero();

      RangeScalar* Y_i = Y;
      Ordinal i = 0;
      for (size_t k = 0; k < nnz; ++k) {
        const MatrixScalar A_ij = Teuchos::ScalarTraits<MatrixScalar>::conjugate (val[k]);
        const Ordinal j = ind[k];
        // NOTE: "if" instead of "while" here is only valid
        // if the matrix contains no empty columns.
        if (k >= ptr[i+1]) {
          // Write temp output from last iteration(s) to Y,
          // before incrementing the current row index.
          if (k > 0) {
            Y_i[0] = tmp[0];
            Y_i[colStrideY] = tmp[1];
            Y_i[2*colStrideY] = tmp[2];
            Y_i[3*colStrideY] = tmp[3];
          }
          ++i;
          // We haven't seen row i before; scale Y(i,:) by beta.
          Y_i = &Y[i];
          tmp[0] = beta * Y_i[0];
          tmp[1] = beta * Y_i[colStrideY];
          tmp[2] = beta * Y_i[2*colStrideY];
          tmp[3] = beta * Y_i[3*colStrideY];
        }
        const DomainScalar* const X_j = &X[j];
        tmp[0] += alpha * A_ij * X_j[0];
        tmp[1] += alpha * A_ij * X_j[colStrideX];
        tmp[2] += alpha * A_ij * X_j[2*colStrideX];
        tmp[3] += alpha * A_ij * X_j[3*colStrideX];
      }
      Y_i[0] = tmp[0];
      Y_i[colStrideY] = tmp[1];
      Y_i[2*colStrideY] = tmp[2];
      Y_i[3*colStrideY] = tmp[3];
    }
  }
}

} // namespace Raw
} // namespace Kokkos

//
// TODO (mfh 26 Jul 2012): Add explicit instantiation macros.
// We'll need one for each function name.
//
#endif // #ifndef __Kokkos_Raw_SparseMatVec_def_hpp
