//@HEADER
// ************************************************************************
// 
//            LOCA: Library of Continuation Algorithms Package
//                 Copyright (2005) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER
                                                                                
// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation 
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u 
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

#include <fstream>

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL}; 

// Finite Element Problem Class
class FiniteElementProblem { 

public:

  // Constructor
  FiniteElementProblem(int NumGlobalElements, Epetra_Comm& Comm, 
		       double s = 1.0, std::ofstream* file = NULL);
  
  // Destructor
  ~FiniteElementProblem();

  // Evaluates the function (F) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate(FillType f, const Epetra_Vector *solnVector, 
		Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix,
		double jac_coeff = 1.0, 
		double mass_coeff = 0.0);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the FiniteElementProblem class.
  Epetra_Vector& getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the FiniteElementProblem class.
  Epetra_CrsMatrix& getJacobian();

  // Set a bifurcation parameter in the application physics
  bool set(std::string label, double value);

  //! Print solution to output file
  void printSolution(const Epetra_Vector& x_, double conParam);

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

  // Private to prohibit copying
  FiniteElementProblem(const FiniteElementProblem&);

  // Private to prohibit copying
  FiniteElementProblem& operator=(const FiniteElementProblem&);

private:

  FillType flag;
  Epetra_Map *StandardMap; 
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Vector *initialSolution;
  Epetra_Vector *rhs;
  Epetra_CrsGraph *AA;
  Epetra_CrsMatrix *A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

 public:

  double factor;          // factor used on nonlinear term
  double leftBC;          // value of the essential BC at x = 0.
  double rightBC;         // value of the essential BC at x = 1.
  double scale;           // scale factor to test parameter scaling

private:

  //! Stores pointer to output file stream
  std::ofstream* outputFile;

  //! Epetra_Map for printing solution to output file
  Epetra_Map* outputMap;

  //! Epetra_Vector for printint solution to output file
  Epetra_Vector* outputVector;

  //! Importer for printing solution to ouput file
  Epetra_Import* outputImporter;

};
#endif




