//@HEADER
// ************************************************************************
// 
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or 
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER
                                                                                
// Finite Element Problem Class
/* Provides function (RHS) and Jacobian evaluations for the  
 * 2 equation test for quadratic and cubic line searches 
 * from Dennis & Schnabel's book, chp 6. example 6.5.1
 *  
 *    U0**2 + U1**2 - 2 = 0
 *    exp(U0-1) + U1**3 -2 = 0
 */

#ifndef _NOX_EXAMPLE_DENNIS_SCHNABEL_651_H
#define _NOX_EXAMPLE_DENNIS_SCHNABEL_651_H

#include "NOX_Epetra_Interface_Required.H"
#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map; 
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType {F_ONLY, MATRIX_ONLY, ALL}; 

// Finite Element Problem Class
class DennisSchnabel { 

public:

  // Constructor
  DennisSchnabel(int NumGlobalElements, Epetra_Comm& Comm);

  // Destructor
  ~DennisSchnabel();

  // Evaluates the function (RHS) and/or the Jacobian using the solution 
  // values in solnVector.
  bool evaluate( NOX::Epetra::Interface::Required::FillType fillType,
                 const Epetra_Vector *solnVector, Epetra_Vector *rhsVector);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by this class.
  Teuchos::RCP<Epetra_Vector> getSolution();
  
  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by this class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraph(Epetra_CrsGraph& AA);

private:

  FillType flag;
  Epetra_Map* StandardMap; 
  Epetra_Map* OverlapMap;
  Epetra_Import* Importer;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector* soln;
  Epetra_Vector* rhs;
  Epetra_CrsGraph* AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm* Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes
  int NumMyElements;      // Number of elements owned by this process
  int NumGlobalElements;  // Total Number of elements

};
#endif




