// Copyright(C) 1999-2010
// Sandia Corporation. Under the terms of Contract
// DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
// certain rights in this software.
//         
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
// 
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//     * Neither the name of Sandia Corporation nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include <Ioss_IOFactory.h>
#include <Ioss_Utils.h>
#include <stddef.h>
#include <map>
#include <ostream>
#include <string>
#include <utility>

#include "Ioss_CodeTypes.h"
#include "Ioss_DBUsage.h"

namespace Ioss {
  class DatabaseIO;
  typedef IOFactoryMap::value_type IOFactoryValuePair;
}  // namespace Ioss


Ioss::DatabaseIO* Ioss::IOFactory::create(const std::string& type,
					  const std::string& filename,
					  Ioss::DatabaseUsage db_usage,
					  MPI_Comm communicator,
					  const Ioss::PropertyManager &properties)
{
  Ioss::DatabaseIO *db = NULL;
  Ioss::IOFactoryMap::iterator iter = registry()->find(type);
  if (iter == registry()->end()) {
    if (registry()->size() == 0) {
      std::ostringstream errmsg;
      errmsg << "ERROR: No database types have been registered.\n"
	     << "       Was Ioss::Init::Initializer() called?\n\n";
      IOSS_ERROR(errmsg);
    } else {
      std::ostringstream errmsg;
      errmsg << "ERROR: The database type '" << type
	     << "' is not supported.\n";
      NameList db_types;
      describe(&db_types);
      errmsg << "\nSupported database types:\n\t";
      for (Ioss::NameList::const_iterator IF = db_types.begin(); IF != db_types.end(); ++IF) {
        errmsg << *IF << "  ";
      }
      errmsg << "\n\n";
      IOSS_ERROR(errmsg);
    }
  } else {
    Ioss::IOFactory* factory = (*iter).second;
    db = factory->make_IO(filename, db_usage, communicator, properties);
  }
  return db;
}

int Ioss::IOFactory::describe(NameList *names)
{
  int count = 0;
  Ioss::IOFactoryMap::const_iterator I;
  for (I = registry()->begin(); I != registry()->end(); ++I) {
    names->push_back((*I).first);
    ++count;
  }
  return count;
}

Ioss::IOFactory::IOFactory(const std::string& type)
{
  registry()->insert(IOFactoryValuePair(type, this));
}

void Ioss::IOFactory::alias(const std::string& base, const std::string& syn)
{
  Ioss::IOFactory* factory = (*registry()->find(base)).second;
  registry()->insert(IOFactoryValuePair(syn, factory));
}

Ioss::IOFactoryMap* Ioss::IOFactory::registry()
{
  static IOFactoryMap registry_;
  return &registry_;
}

void Ioss::IOFactory::clean()
{
}
