// @HEADER
//
// ***********************************************************************
//
//             Xpetra: A linear algebra interface package
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jeremie Gaidamour (jngaida@sandia.gov)
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//
// ***********************************************************************
//
// @HEADER
#ifndef XPETRA_EPETRACRSGRAPH_HPP
#define XPETRA_EPETRACRSGRAPH_HPP

/* this file is automatically generated - do not edit (see script/epetra.py) */

#include "Xpetra_EpetraConfigDefs.hpp"

#include "Xpetra_CrsGraph.hpp"

#include "Xpetra_EpetraMap.hpp"
#include "Xpetra_EpetraImport.hpp"
#include "Xpetra_EpetraUtils.hpp"

#include <Epetra_CrsGraph.h>

namespace Xpetra {

  // TODO: move that elsewhere
  RCP< const CrsGraph<int, int> > toXpetra(const Epetra_CrsGraph& graph);
  const Epetra_CrsGraph & toEpetra(const RCP< const CrsGraph<int, int> > &graph);
  //

#ifndef DOXYGEN_SHOULD_SKIP_THIS
  // forward declaration
  template <class S, class LO, class GO, class N, class SpMatOps>
  class CrsMatrix;
#endif

  class EpetraCrsGraph
    : public CrsGraph<int, int>
  {

    typedef int LocalOrdinal;
    typedef int GlobalOrdinal;
    typedef Kokkos::DefaultNode::DefaultNodeType Node;
    //! The specialization of Map used by this class.
    typedef Map<LocalOrdinal,GlobalOrdinal,Node> map_type;

  public:

    //! @name Constructor/Destructor Methods
    //@{

    //! Constructor specifying fixed number of entries for each row.
    EpetraCrsGraph(const RCP< const map_type > &rowMap, size_t maxNumEntriesPerRow, ProfileType pftype=DynamicProfile, const RCP< ParameterList > &params=null);

    //! Constructor specifying (possibly different) number of entries in each row.
    EpetraCrsGraph(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rowMap, const ArrayRCP< const size_t > &NumEntriesPerRowToAlloc, ProfileType pftype=DynamicProfile, const RCP< ParameterList > &params=null);

    //! Constructor specifying column Map and fixed number of entries for each row.
    EpetraCrsGraph(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rowMap, const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &colMap, size_t maxNumEntriesPerRow, ProfileType pftype=DynamicProfile, const RCP< ParameterList > &params=null);

    //! Constructor specifying column Map and number of entries in each row.
    EpetraCrsGraph(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rowMap, const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &colMap, const ArrayRCP< const size_t > &NumEntriesPerRowToAlloc, ProfileType pftype=DynamicProfile, const RCP< ParameterList > &params=null);

    //! Destructor.
    virtual ~EpetraCrsGraph() { }

    //@}

    //! @name Insertion/Removal Methods
    //@{

    //! Insert graph indices, using global IDs.
    void insertGlobalIndices(GlobalOrdinal globalRow, const ArrayView< const GlobalOrdinal > &indices);

    //! Insert graph indices, using local IDs.
    void insertLocalIndices(LocalOrdinal localRow, const ArrayView< const LocalOrdinal > &indices);

    //! Remove all graph indices from the specified local row.
    void removeLocalIndices(LocalOrdinal localRow) { XPETRA_MONITOR("EpetraCrsGraph::removeLocalIndices"); graph_->RemoveMyIndices(localRow); }

    //@}

    //! @name Transformational Methods
    //@{

    //! Signal that data entry is complete, specifying domain and range maps.
    void fillComplete(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &domainMap, const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rangeMap, const RCP< ParameterList > &params=null);

    //! Signal that data entry is complete.
    void fillComplete(const RCP< ParameterList > &params=null);

    //@}

    //! @name Methods implementing RowGraph.
    //@{

    //! Returns the communicator.
    const RCP< const Comm< int > >  getComm() const { XPETRA_MONITOR("EpetraCrsGraph::getComm"); return toXpetra(graph_->Comm()); }

    //! Returns the Map that describes the row distribution in this graph.
    const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > >  getRowMap() const { XPETRA_MONITOR("EpetraCrsGraph::getRowMap"); return toXpetra(graph_->RowMap()); }

    //! Returns the Map that describes the column distribution in this graph.
    const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > >  getColMap() const { XPETRA_MONITOR("EpetraCrsGraph::getColMap"); return toXpetra(graph_->ColMap()); }

    //! Returns the Map associated with the domain of this graph.
    const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > >  getDomainMap() const { XPETRA_MONITOR("EpetraCrsGraph::getDomainMap"); return toXpetra(graph_->DomainMap()); }

    //! Returns the Map associated with the domain of this graph.
    const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > >  getRangeMap() const { XPETRA_MONITOR("EpetraCrsGraph::getRangeMap"); return toXpetra(graph_->RangeMap()); }

    //! Returns the importer associated with this graph.
    RCP< const Import< LocalOrdinal, GlobalOrdinal, Node > > getImporter() const { XPETRA_MONITOR("EpetraCrsGraph::getImporter"); return toXpetra(graph_->Importer()); }

    //! Returns the number of global rows in the graph.
    global_size_t getGlobalNumRows() const { XPETRA_MONITOR("EpetraCrsGraph::getGlobalNumRows"); return graph_->NumGlobalRows(); }

    //! Returns the number of global columns in the graph.
    global_size_t getGlobalNumCols() const { XPETRA_MONITOR("EpetraCrsGraph::getGlobalNumCols"); return graph_->NumGlobalCols(); }

    //! Returns the number of graph rows owned on the calling node.
    size_t getNodeNumRows() const { XPETRA_MONITOR("EpetraCrsGraph::getNodeNumRows"); return graph_->NumMyRows(); }

    //! Returns the number of columns connected to the locally owned rows of this graph.
    size_t getNodeNumCols() const { XPETRA_MONITOR("EpetraCrsGraph::getNodeNumCols"); return graph_->NumMyCols(); }

    //! Returns the index base for global indices for this graph.
    GlobalOrdinal getIndexBase() const { XPETRA_MONITOR("EpetraCrsGraph::getIndexBase"); return graph_->IndexBase(); }

    //! Returns the global number of entries in the graph.
    global_size_t getGlobalNumEntries() const { XPETRA_MONITOR("EpetraCrsGraph::getGlobalNumEntries"); return graph_->NumGlobalEntries(); }

    //! Returns the local number of entries in the graph.
    size_t getNodeNumEntries() const { XPETRA_MONITOR("EpetraCrsGraph::getNodeNumEntries"); return graph_->NumMyEntries(); }

    //! Returns the current number of entries on this node in the specified global row.
    size_t getNumEntriesInGlobalRow(GlobalOrdinal globalRow) const { XPETRA_MONITOR("EpetraCrsGraph::getNumEntriesInGlobalRow"); return graph_->NumGlobalIndices(globalRow); }

    //! Returns the current number of entries on this node in the specified local row.
    size_t getNumEntriesInLocalRow(LocalOrdinal localRow) const { XPETRA_MONITOR("EpetraCrsGraph::getNumEntriesInLocalRow"); return graph_->NumMyIndices(localRow); }

    //! Returns the current number of allocated entries for this node in the specified global row .
    size_t getNumAllocatedEntriesInGlobalRow(GlobalOrdinal globalRow) const { XPETRA_MONITOR("EpetraCrsGraph::getNumAllocatedEntriesInGlobalRow"); return graph_->NumAllocatedGlobalIndices(globalRow); }

    //! Returns the current number of allocated entries on this node in the specified local row.
    size_t getNumAllocatedEntriesInLocalRow(LocalOrdinal localRow) const { XPETRA_MONITOR("EpetraCrsGraph::getNumAllocatedEntriesInLocalRow"); return graph_->NumAllocatedMyIndices(localRow); }

    //! Returns the number of global diagonal entries, based on global row/column index comparisons.
    global_size_t getGlobalNumDiags() const { XPETRA_MONITOR("EpetraCrsGraph::getGlobalNumDiags"); return graph_->NumGlobalDiagonals(); }

    //! Returns the number of local diagonal entries, based on global row/column index comparisons.
    size_t getNodeNumDiags() const { XPETRA_MONITOR("EpetraCrsGraph::getNodeNumDiags"); return graph_->NumMyDiagonals(); }

    //! Returns the maximum number of entries across all rows/columns on all nodes.
    size_t getGlobalMaxNumRowEntries() const { XPETRA_MONITOR("EpetraCrsGraph::getGlobalMaxNumRowEntries"); return graph_->GlobalMaxNumIndices(); }

    //! Returns the maximum number of entries across all rows/columns on this node.
    size_t getNodeMaxNumRowEntries() const { XPETRA_MONITOR("EpetraCrsGraph::getNodeMaxNumRowEntries"); return graph_->MaxNumIndices(); }

    //! Indicates whether the graph has a well-defined column map.
    bool hasColMap() const { XPETRA_MONITOR("EpetraCrsGraph::hasColMap"); return graph_->HaveColMap(); }

    //! Indicates whether the graph is lower triangular.
    bool isLowerTriangular() const { XPETRA_MONITOR("EpetraCrsGraph::isLowerTriangular"); return graph_->LowerTriangular(); }

    //! Indicates whether the graph is upper triangular.
    bool isUpperTriangular() const { XPETRA_MONITOR("EpetraCrsGraph::isUpperTriangular"); return graph_->UpperTriangular(); }

    //! If graph indices are in the local range, this function returns true. Otherwise, this function returns false. */.
    bool isLocallyIndexed() const { XPETRA_MONITOR("EpetraCrsGraph::isLocallyIndexed"); return graph_->IndicesAreLocal(); }

    //! If graph indices are in the global range, this function returns true. Otherwise, this function returns false. */.
    bool isGloballyIndexed() const { XPETRA_MONITOR("EpetraCrsGraph::isGloballyIndexed"); return graph_->IndicesAreGlobal(); }

    //! Returns true if fillComplete() has been called and the graph is in compute mode.
    bool isFillComplete() const { XPETRA_MONITOR("EpetraCrsGraph::isFillComplete"); return graph_->Filled(); }

    //! Returns true if storage has been optimized.
    bool isStorageOptimized() const { XPETRA_MONITOR("EpetraCrsGraph::isStorageOptimized"); return graph_->StorageOptimized(); }

    //! Extract a const, non-persisting view of global indices in a specified row of the graph.
    void getGlobalRowView(GlobalOrdinal GlobalRow, ArrayView< const GlobalOrdinal > &Indices) const;

    //! Extract a const, non-persisting view of local indices in a specified row of the graph.
    void getLocalRowView(LocalOrdinal LocalRow, ArrayView< const LocalOrdinal > &indices) const;

    //@}

    //! @name Overridden from Teuchos::Describable
    //@{

    //! Return a simple one-line description of this object.
    std::string description() const;

    //! Print the object with some verbosity level to an FancyOStream object.
    void describe(Teuchos::FancyOStream &out, const Teuchos::EVerbosityLevel verbLevel=Teuchos::Describable::verbLevel_default) const;

    //@}

    //! Implements DistObject interface
    //{@

    //! Access function for the Tpetra::Map this DistObject was constructed with.
    const Teuchos::RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > getMap() const { XPETRA_MONITOR("EpetraCrsGraph::getMap"); return toXpetra(graph_->Map()); }

    //! Import.
    void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source, const Import< LocalOrdinal, GlobalOrdinal, Node > &importer, CombineMode CM);

    //! Export.
    void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest, const Import< LocalOrdinal, GlobalOrdinal, Node >& importer, CombineMode CM);

    //! Import (using an Exporter).
    void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source, const Export< LocalOrdinal, GlobalOrdinal, Node >& exporter, CombineMode CM);

    //! Export (using an Importer).
    void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest, const Export< LocalOrdinal, GlobalOrdinal, Node >& exporter, CombineMode CM);

    //@}

    //! @name Xpetra specific
    //@{

    //! EpetraCrsGraph constructor to wrap a Epetra_CrsGraph object
    EpetraCrsGraph(const Teuchos::RCP<Epetra_CrsGraph> &graph) : graph_(graph) { }

    //! Get the underlying Epetra graph
    RCP< const Epetra_CrsGraph> getEpetra_CrsGraph() const { return graph_; }

    //@}

  private:

    RCP<Epetra_CrsGraph> graph_;

  }; // EpetraCrsGraph class

} // Xpetra namespace

#endif // XPETRA_EPETRACRSGRAPH_HPP
