import asyncio
import os
import socketio
import sys
import argparse
import pytz
from collections import defaultdict
from datetime import datetime, timedelta
from src.telemetry import ( telemetry_response, get_stream_data )
from src.commanding import commanding_response
from src.external_data import get_item_response, search_items_response
from src.notifications import notifications_response

timezone = pytz.timezone("UTC")

EPSILON3_API_URL = os.getenv('EPSILON3_API_URL', 'https://api.epsilon3.io')
print(f"Using EPSILON3_API_URL {EPSILON3_API_URL}")

# Set an environment variable with your API key.
# Warning! Always keep your key secret and never reveal it publicly or store it
# in code repositories.
EPSILON3_API_KEY = os.getenv('EPSILON3_API_KEY')
if not EPSILON3_API_KEY:
  sys.exit("Must set EPSILON3_API_KEY environment variable.")

# Create socket.io client
io = socketio.AsyncClient(logger=True)

@io.event
async def connect():
    '''Server connection event handler.'''
    print(f'Connected to Epsilon3 API at {EPSILON3_API_URL}')

@io.on('get_sample', namespace='/v1/telemetry/realtime')
async def get_sample(parameter):
    '''
    Parameter sample (value) request handler.

    Args:
        parameter (dictionary): A parameter object.

    Returns:
        sample (dictionary): A sample object with the latest parameter value.
    '''
    task = asyncio.create_task(telemetry_response(parameter))
    return await task

@io.on('send_command', namespace='/v1/commands/realtime')
async def send_command(command):
    '''
    Send command request handler. Handles both sync and async commands.

    Args:
        command (dictionary): A command object. If it has a correlation_id, it will be handled asynchronously.

    Returns:
        results (dictionary): A command results object (sync) or immediate response (async).
    '''
    if command.get('correlation_id'): # Return an immediate "executing" response for async commands
        async_response = {
            'status': 'executing',
            'received_at': datetime.now(timezone).strftime('%Y-%m-%dT%H:%M:%SZ'),
            'correlation_id': command['correlation_id']
        }

        asyncio.create_task(execute_command_async(command))
        return async_response
    else: # Handle synchronous command
        task = asyncio.create_task(commanding_response(command))
        return await task

@io.on('item', namespace='/v1/external-data/items')
async def get_item(request):
    '''
    Finds the item with the given item type and id

    Args:
        request (dictionary): A request for an item

    Returns:
        results (dictionary): An item object
    '''
    task = asyncio.create_task(get_item_response(request))
    return await task

@io.on('search', namespace='/v1/external-data/items')
async def search_items(request):
    '''
    Searches external data items with a given search term.

    Args:
        request (dictionary): A search request.

    Returns:
        results (dictionary): A search results object.
    '''
    task = asyncio.create_task(search_items_response(request))
    return await task

@io.on('notification', namespace='/v1/notifications/realtime')
async def log_notification(request):
    '''
    Logs notifications to console.

    Args:
        request (dictionary): A notification event.

    Returns:
        results (dictionary): An acknowledgement the notification was received.
    '''
    task = asyncio.create_task(notifications_response(request))
    return await task

# listeners to enable bulk streaming in runs

streamed_telemetry = defaultdict(int)

@io.on('start_stream', namespace='/v1/telemetry/realtime')
async def on_start_telemetry_streaming(request):
    param_name = request['name']
    global streamed_telemetry
    stream_id = request['stream_id']
    streamed_telemetry[stream_id] += 1

    if (streamed_telemetry[stream_id] > 1):
        return

    asyncio.create_task(stream_telemetry(dict(
            name=param_name,
            operation=request.get('operation'),
            variables=request.get('variables'),
            metadata=request.get('metadata'),
            dictionary_id=request.get('dictionary_id')
        ),
        stream_id,
        request.get('refresh_rate') or 1,
    ))

@io.on('end_streams', namespace='/v1/telemetry/realtime')
async def on_end_telemetry_streaming(request):
    global streamed_telemetry
    for stream_id in request['stream_ids']:
        streamed_telemetry.pop(stream_id, None)

@io.on('disconnect', namespace='/v1/telemetry/realtime')
async def on_disconnect_telemetry_stream():
    global streamed_telemetry
    streamed_telemetry = defaultdict(int)

async def stream_telemetry(payload, stream_id, refresh_rate):
    global streamed_telemetry
    while streamed_telemetry[stream_id] > 0:
        data = await telemetry_response(payload)
        response = {
            'data': data,
            'stream_id': stream_id,
        }
        await io.emit('data_update', response, namespace='/v1/telemetry/realtime')
        await asyncio.sleep(refresh_rate)

# listeners to enable bulk streaming independent of runs

@io.on('connect', namespace='/v1/telemetry/realtime')
async def on_connect_telemetry_namespace():
    parser = argparse.ArgumentParser()
    parser.add_argument('-stream', action='store_true', help='An optional argument to enable streaming telemetry')
    args = parser.parse_args()

    if args.stream:
        await init_stream_data()

stream_data_enabled = False

async def handle_init_request(request):
    global stream_data_enabled
    stream_data_enabled = True
    asyncio.create_task(stream_data())

@io.on('error', namespace='/v1/telemetry/realtime')
async def handle_error_request(request):
    print('error', request)
    global stream_data_enabled
    stream_data_enabled = False

# To receive information about dangling streams and exceeded rate limits
@io.on('telemetry_connection_status', namespace='/v1/telemetry/realtime')
async def handle_connection_status(request):
    print('status', request)

async def stream_data():
    while stream_data_enabled:
        await asyncio.sleep(1)  # Adjust the frequency of data streaming as needed
        data = await get_stream_data()
        data['end_time']= (datetime.now(timezone) + timedelta(seconds=60)).isoformat()
        await io.emit('stream_update', data, namespace='/v1/telemetry/realtime')

async def init_stream_data():
    await asyncio.sleep(0.1) # Short delay to allow for namespace connections to finish
    data = {'end_time': (datetime.now(timezone) + timedelta(seconds=60)).isoformat()}
    await io.emit('init', data, namespace='/v1/telemetry/realtime', callback=handle_init_request)

async def execute_command_async(command):
    '''
    Execute the actual command and emit the finished response
    '''
    try:
        result = await commanding_response(command)

        if result is not None:
            finished_response = {
                'status': 'completed',
                'correlation_id': command['correlation_id'],
                'results': result,
                'completed_at': datetime.now(timezone).strftime('%Y-%m-%dT%H:%M:%SZ'),
            }

            await io.emit('command_finished', finished_response, namespace='/v1/commands/realtime')
            print(f"Async command {command['correlation_id']} completed")

    except Exception as e:
        error_response = {
            'status': 'failed',
            'correlation_id': command['correlation_id'],
            'completed_at': datetime.now(timezone).strftime('%Y-%m-%dT%H:%M:%SZ'),
        }
        await io.emit('command_finished', error_response, namespace='/v1/commands/realtime')
        print(f"Async command {command['correlation_id']} failed: {str(e)}")

async def main():
    '''
    Main app entry point. Connects to the Epsilon3 API and responds to realtime
    requests for telemetry parameter data.
    '''

    # Use API key for authentication.
    auth = {
        'key': EPSILON3_API_KEY
    }

    # Start realtime client connection and enter asyncio event loop.
    await io.connect(EPSILON3_API_URL, auth=auth, transports=["websocket"])

    await io.wait()

asyncio.run(main())
