"""
This sample client responds to a few commands that perform no real actions.

To respond to your own commands simply add/modify an `if` statement with the name of the command
created in Epsilon3 and custom logic to respond to it. The `message` parameter is displayed on the
Epsilon3 UI as plain text and `success` is used as a failure/success (red/green) visual indicator on the UI.

For more information on CRUD operations for commands see https://docs.epsilon3.io/#commanding-api
"""

import base64
from datetime import datetime
from PIL import Image, ImageShow
import asyncio

async def commanding_response(command):
    print('Send command requested:', command)

    # Check command name and send command.
    success = False
    message = ''
    if (command['name'] == 'test.shuttle.image'):
        # use await for long running action (e.g. await my_long_function(param)) to avoid blocking other calls
        im = Image.open("src/sample_data/shuttle_launch.jpg")
        success = ImageShow.show(im) == 1  # Returns 0 on fail, 1 on success
    if (command['name'] == 'sample.async.command'):
        await asyncio.sleep(30)
        success = True
        message = 'Sample async command completed'
    elif (command['name'] == 'test.color'):
        red = float(command['arguments']['red'])
        green = float(command['arguments']['green'])
        blue = float(command['arguments']['blue'])
        print(f'Set color with red: {red} green: {green} blue: {blue}')
        success = True
    elif (command['name'] == 'test.gauge.state'):
        state = int(command['arguments']['state'])
        if (state == 0):
            print('Set gauge with state: OFF')
        if (state == 1):
            print('Set gauge with state: ON')
        success = True
    elif (command['name'] == 'test.config.file'):
        # Decode file contents into binary
        data = base64.b64decode(str(command['arguments']['config.file']['data_base64']))
        print(f'Parsed file contents with length: {len(data)}')
        success = True
        message = 'File received'
    else:
        return None

    # Use current time for received_at value, in ISO 8601 format.
    received_at_utc = datetime.utcnow()
    received_at_str = received_at_utc.strftime('%Y-%m-%dT%H:%M:%SZ')

    # Return command results object.
    response = {
        'name': command['name'],
        'success': success,
        'message': message,
        'received_at': received_at_str
    }
    return response
