"""
This sample client responds to a few external data queries with sample data that illustrates both the
search by term and retrieve functionality.

To respond to your own search queries simply add/modify an `if` statement with the name of the external data type
created in Epsilon3 and custom logic to respond to it. The `items` should contain the response as an array of objects.

For more information on CRUD operations for external data types see https://docs.epsilon3.io/#external-data-api
"""

from .sample_data.external_data_items import (
    WORK_ORDERS,
    PART_NUMBERS,
    SUPPORT_TICKETS,
    PLACES,
)
import requests
import os
import json

FLOW_API_KEY = os.getenv('FLOW_API_KEY')
FLOW_ORGANIZATION_ID = os.getenv('FLOW_ORGANIZATION_ID')
FLOW_PROJECT_REQUIREMENTS_API_URL = os.getenv('FLOW_API_URL', 'https://api.flowengineering.com/rest/v1/org/{orgSlug}/project/{projectSlug}/requirements')

async def get_item_response(request):
    print('External data item requested', request)
    request_type = request['type']
    id = request['id']

    success = True
    message = ''
    items = []
    if (request_type == 'work_order'):
        items = WORK_ORDERS
    elif (request_type == 'part_number'):
        items = PART_NUMBERS
    elif (request_type == 'support_ticket'):
        items = SUPPORT_TICKETS
    elif (request_type == 'place'):
        items = PLACES
    else:
        return None

    item, success = _get_item(items, id)

    # Return item object
    response = {
        'type': request_type,
        'success': success,
        'message': message,
        'item': item,
    }
    return response

async def search_items_response(request):
    print('External data search requested:', request)
    request_type = request['type']
    term = request['term']

    # Check command name and send command.
    success = True
    message = ''
    items = []
    if (request_type == 'work_order'):
        items = WORK_ORDERS
    elif (request_type == 'part_number'):
        items = PART_NUMBERS
    elif (request_type == 'support_ticket'):
        items = SUPPORT_TICKETS
    elif (request_type == 'place'):
        items = PLACES
    elif (request_type.startswith('flow_requirements.')):
        # This portion adds integration with Flow Requirements Software, for more information see the link
        # https://support.epsilon3.io/en/articles/9522784-flow-integration
        project = request_type.split('.')[1]
        url = FLOW_PROJECT_REQUIREMENTS_API_URL.format(orgSlug=FLOW_ORGANIZATION_ID, projectSlug=project)
        response = requests.get(url, headers={"Authorization": FLOW_API_KEY})
        requirements = response.json()
        for requirement in requirements:
            items.append({
                'id': requirement['id'],
                'name': requirement['name'],
                'label': requirement['name'],
                'valid' : True,
                'details': [{
                    'name': 'Requirement Name',
                    'value': requirement['name']
                },
                {
                    'name': 'Requirement ID',
                    'value': requirement['id']
                },
                {
                    'name': 'Requirement Statement',
                    'value': extract_text(json.loads(requirement['statement_raw'] or '{}'))
                }]
            })
    else:
        print('Unrecognized data type:', request_type)
        message = 'Unrecognized data type'
        success = False

    # Filter items by search term.
    items = [item for item in items if _contains_term(item, term)]

    # Return search results object.
    response = {
        'type': request_type,
        'success': success,
        'message': message,
        'items': items,
    }
    return response

def extract_text(requirement_statement):
  results = ''

  for child in requirement_statement:
    if child.get('text'):
      results += child.get('text') + '\n'
    if child.get('children'):
      results += extract_text(child.get('children'))
  return results

def _get_item(items, id):
  '''
  Finds the item with the given id
  '''
  id = id.lower()
  for item in items:
    if item['id'].lower() == id:
      return item, True
  return None, False

def _contains_term(item, term):
    '''
    Returns true if item contains the search term.
    '''
    lowercased = term.lower()
    if 'label' in item and item['label'].lower().find(lowercased) != -1:
        return True
    if item['id'].lower().find(lowercased) != -1:
        return True
    if item['name'].lower().find(lowercased) != -1:
        return True
    return False
