"""
This client provides examples of how to handle different telemetry parameter types.
To respond to your own telemetry queries, add your own code `if` statement with the name of the parameter
created in Epsilon3 and custom logic to respond to it assigned to the variable `value` or `values` (for bulk
telemetry). The `stale_after_ms` parameter is used as a visual indicator to signal when a value is stale on the UI.

For more information on CRUD operations for telemetry parameters see https://docs.epsilon3.io/#telemetry-api
"""

import random
import time
from datetime import datetime, timezone

async def telemetry_response(parameter):
    print('Parameter sample requested:', parameter['name'])
    value = None
    values = None

    #### EXAMPLE CODE, REPLACE WITH YOUR CUSTOM CODE
    # Respond to a telemetry parameter named 'iss.position.latitude' with a random value
    if (parameter['name'] == 'iss.position.latitude'):
        value = random.random() * -100

    # You can access other fields from the parameter, like procedure `variables`
    # See https://docs.epsilon3.io/#send-telemetry-values-to-epsilon3 for more
    elif (parameter['name'] == 'iss.position.longitude'):
        multiplier = parameter['variables'].get('sample_procedure_variable') or 1
        value = random.random() * multiplier * 100

    elif (parameter['name'] == 'iss.position.latitude_sign'):
        value = 0 if random.randint(-5, 5) < 0 else 1

    # Response to an aggregate telemetry parameter called 'iss.position'
    elif (parameter['name'] == 'iss.position'):
        value = {
            'latitude': random.random() * 100,
            'latitude_sign': 0 if random.randint(-5, 5) < 0 else 1,
            'longitude': random.random() * -100,
            'longitude_sign': 0 if random.randint(-5, 5) < 0 else 1,
        }

    # You can send a list of values as a batch for more detail than the polling interval
    # The batch data items should be an object with `value` and `timestamp` fields
    # All items are stored for the data module but only the first is displayed in the run
    elif (parameter['name'] == 'iss.batch.position.latitude'):
        timestamp = time.time() * 1000
        values = [
            {
                "value": value,
                "timestamp": timestamp,
            },
            {
                "value": value,
                "timestamp": timestamp + 200,
            },
        ]

    # This is a catch-all if no parameter name matched
    # If you have multiple connected clients returning an empty response
    # gives other clients the chance to respond
    else:
        print('Unhandled parameter name:', parameter['name'])
        return

    # Data must be returned to Epsilon 3 in this format. It must be an object with
    # 'name', 'recorded_at', 'stale_after_ms' and 'value' OR 'values' as fields
    sample = {
        'name': parameter['name'],
        'recorded_at': datetime.now(timezone.utc).strftime('%Y-%m-%dT%H:%M:%SZ'),
        'stale_after_ms': 10000 # Period after which a telemetry sample is considered stale (ms)
    }

    # Respond with either a single value or a bulk values upload, not both
    if value is not None:
        sample['value'] = value
    elif values is not None:
        sample['values'] = values
    else:
        print('Error processing telemetry request, please specify one of `value` or `values`', parameter['name'])

    return sample


# This is only used for pushing data to Epsilon3's data module with the option `-stream` enabled on script startup.
# Not used for telemetry polling from runs
async def get_stream_data():
    timestamp = time.time() * 1000
    formatted_data = [
        {
            "id": 1,
            "data": [
                {
                    "timestamp": timestamp * 1000,
                    "value": random.random() * 100
                },
            ]
        },
        {
            "id": 2,
            "data": [
                {
                    "timestamp": timestamp * 1000,
                    "value": random.random() * -100
                },
            ]
        }
    ]
    return formatted_data
