import os
import sys
import asyncio
import requests
from datetime import datetime
from aioflask import Flask, request, jsonify
from src.telemetry import telemetry_response
from src.commanding import commanding_response
from src.external_data import get_item_response, search_items_response
from src.notifications import notifications_response
from utils.webhook_auth import verify_webhook_signature

EPSILON3_HEADER = 'Epsilon3-Signature'

# Warning! Always keep your key secret and never reveal it publicly or store it
# in code repositories.
EPSILON3_WEBHOOK_KEY = os.getenv('EPSILON3_WEBHOOK_KEY')
if not EPSILON3_WEBHOOK_KEY:
  sys.exit("Must set EPSILON3_WEBHOOK_KEY environment variable.")

EPSILON3_API_URL = os.getenv('EPSILON3_API_URL', 'https://api.epsilon3.io')
# Warning! Always keep your key secret and never reveal it publicly or store it
# in code repositories.
EPSILON3_API_KEY = os.getenv('EPSILON3_API_KEY')

app = Flask(__name__)

@app.route('/', methods=['POST'])
async def webhook():
    '''
    Endpoint for handling requests sent from the Epsilon3 app.

    Examples: telemetry, commanding, external data, etc.

    Args:
        request (dictionary): A request from Epsilon3.

    Returns:
        results (dictionary): A response sent back to Epsilon3.
    '''

    payload = request.get_json()

    key = EPSILON3_WEBHOOK_KEY
    signature = request.headers[EPSILON3_HEADER]
    verified = verify_webhook_signature(key, payload, signature)

    response = None

    if not verified:
        print('Request signature could not be verified')
        return jsonify(response)

    if (payload['type'] == 'telemetry.get_sample'):
        response = await telemetry_response(payload['data'])
    elif (payload['type'] == 'commands.send_command'):
        command_data = payload['data']
        if command_data.get('correlation_id'): # Return an immediate "executing" response for async commands
            if not EPSILON3_API_KEY:
                sys.exit("Must set EPSILON3_API_KEY environment variable.")
            response = {
                'status': 'executing',
                'received_at': datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'),
                'correlation_id': command_data['correlation_id']
            }
            asyncio.create_task(execute_command_async(command_data))
        else: # Handle synchronous commands
            response = await commanding_response(command_data)
    elif (payload['type'] == 'external-data.item'):
        response = get_item_response(payload['data'])
    elif (payload['type'] == 'external-data.search'):
        response = search_items_response(payload['data'])
    elif (payload['type'] == 'notifications.notification'):
        response = notifications_response(payload['data'])

    return jsonify(response)

async def execute_command_async(command):
    '''
    Execute the actual command and send completion back to Epsilon3 via restAPI
    '''
    try:
        result = await commanding_response(command)

        if result is not None:
            finished_response = {
                'status': 'completed',
                'correlation_id': command['correlation_id'],
                'results': result,
                'completed_at': datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ')
            }

            await send_command_completion(finished_response)
            print(f"Async webhook command {command['correlation_id']} completed")

    except Exception as e:
        error_response = {
            'status': 'failed',
            'correlation_id': command['correlation_id'],
            'completed_at': datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%SZ'),
        }
        await send_command_completion(error_response)
        print(f"Async webhook command {command['correlation_id']} failed: {str(e)}")

async def send_command_completion(response):
    '''
    Send command completion back to Epsilon3 API via restAPI
    '''
    try:
        url = f"{EPSILON3_API_URL}/v1/commands/webhook-command-finished"
        api_response = requests.post(url, auth=(EPSILON3_API_KEY, ''), json=response)
        if api_response.status_code == 200:
            print(f"Successfully sent completion for {response['correlation_id']}")
        else:
            print(f"Failed to send completion: {api_response.status_code} - {api_response.text}")

    except Exception as e:
        print(f"Error sending command completion: {str(e)}")
