/*
 * Decompiled with CFR 0.152.
 */
package com.android.helper.aoa;

import com.android.helper.aoa.AoaHID;
import com.android.helper.aoa.AoaKey;
import com.android.helper.aoa.UsbDevice;
import com.android.helper.aoa.UsbException;
import com.android.helper.aoa.UsbHelper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Range;
import com.google.common.util.concurrent.Uninterruptibles;
import java.awt.Point;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import javax.annotation.Nonnull;

public class AoaDevice
implements AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(AoaDevice.class.getName());
    static final int DEVICE_NOT_FOUND = -4;
    static final byte INPUT = -64;
    static final byte OUTPUT = 64;
    static final int GOOGLE_VID = 6353;
    private static final Range<Integer> AOA_PID = Range.closed(11520, 11525);
    private static final Range<Integer> AUDIO_PID = Range.closed(11522, 11525);
    private static final ImmutableSet<Integer> ADB_PID = ImmutableSet.of(Integer.valueOf(11521), Integer.valueOf(11523), Integer.valueOf(11525));
    private static final byte[] MANUFACTURER = "Android\u0000".getBytes(Charsets.UTF_8);
    private static final byte[] MODEL = (AoaDevice.class.getName() + "\u0000").getBytes(Charsets.UTF_8);
    private static final byte[] VERSION = "1.0\u0000".getBytes(Charsets.UTF_8);
    static final byte ACCESSORY_GET_PROTOCOL = 51;
    static final byte ACCESSORY_SEND_STRING = 52;
    static final byte ACCESSORY_START = 53;
    static final byte ACCESSORY_REGISTER_HID = 54;
    static final byte ACCESSORY_UNREGISTER_HID = 55;
    static final byte ACCESSORY_SET_HID_REPORT_DESC = 56;
    static final byte ACCESSORY_SEND_HID_EVENT = 57;
    static final byte ACCESSORY_SET_AUDIO_MODE = 58;
    static final int ACCESSORY_START_MAX_RETRIES = 5;
    static final byte TOUCH_UP = 0;
    static final byte TOUCH_DOWN = 3;
    static final byte SYSTEM_WAKE = 1;
    static final byte SYSTEM_HOME = 2;
    static final byte SYSTEM_BACK = 4;
    private static final Duration CONNECTION_TIMEOUT = Duration.ofSeconds(10L);
    private static final Duration CONFIGURE_DELAY = Duration.ofSeconds(1L);
    private static final Duration ACTION_DELAY = Duration.ofSeconds(3L);
    private static final Duration STEP_DELAY = Duration.ofMillis(10L);
    static final Duration LONG_CLICK = Duration.ofSeconds(1L);
    private final UsbHelper mHelper;
    private UsbDevice mDelegate;
    private final String mSerialNumber;

    AoaDevice(@Nonnull UsbHelper helper, @Nonnull UsbDevice delegate2) {
        this.mHelper = helper;
        this.mDelegate = delegate2;
        if (!this.isValid()) {
            throw new UsbException("Invalid device connection", new Object[0]);
        }
        this.mSerialNumber = this.mDelegate.getSerialNumber();
        if (this.mSerialNumber == null) {
            throw new UsbException("Could not determine device serial number", new Object[0]);
        }
        this.initialize();
    }

    private void initialize() {
        int attempt = 0;
        while (true) {
            if (this.isAccessoryMode()) {
                this.registerHIDs();
                return;
            }
            if (attempt >= 5) {
                LOGGER.warning(String.format("Failed to start accessory mode on %s after %d attempts; proceeding anyway", this.mSerialNumber, attempt));
                this.registerHIDs();
                return;
            }
            this.transferOrThrow((byte)52, 0, 0, MANUFACTURER);
            this.transferOrThrow((byte)52, 0, 1, MODEL);
            this.transferOrThrow((byte)52, 0, 3, VERSION);
            this.transferOrThrow((byte)58, 1, 0, new byte[0]);
            this.transferOrThrow((byte)53, 0, 0, new byte[0]);
            this.sleep(CONFIGURE_DELAY);
            this.mDelegate.close();
            this.reconnect();
            ++attempt;
        }
    }

    private int transfer(byte request, int value, int index, byte[] data) {
        return this.mDelegate.controlTransfer((byte)64, request, value, index, data);
    }

    private void transferOrThrow(byte request, int value, int index, byte[] data) {
        this.mHelper.checkResult(this.transfer(request, value, index, data));
    }

    private void reconnect() {
        this.mDelegate = this.mHelper.getDevice(this.mSerialNumber, CONNECTION_TIMEOUT);
        if (!this.isValid()) {
            throw new UsbException("Timed out while reconnecting to device %s", this.mSerialNumber);
        }
    }

    private void registerHIDs() {
        for (AoaHID hid : AoaHID.values()) {
            this.transferOrThrow((byte)54, hid.getId(), hid.getDescriptor().length, new byte[0]);
            this.transferOrThrow((byte)56, hid.getId(), 0, hid.getDescriptor());
        }
        this.sleep(CONFIGURE_DELAY);
    }

    private void unregisterHIDs() {
        for (AoaHID hid : AoaHID.values()) {
            this.transfer((byte)55, hid.getId(), 0, new byte[0]);
        }
    }

    public void resetConnection() {
        this.close();
        this.reconnect();
        this.initialize();
    }

    public boolean isValid() {
        return this.mDelegate != null && this.mDelegate.isValid();
    }

    @Nonnull
    public String getSerialNumber() {
        return this.mSerialNumber;
    }

    private boolean isAccessoryMode() {
        return 6353 == this.mDelegate.getVendorId() && AOA_PID.contains(this.mDelegate.getProductId());
    }

    public boolean isAdbEnabled() {
        return 6353 == this.mDelegate.getVendorId() && ADB_PID.contains(this.mDelegate.getProductId());
    }

    public boolean isAudioEnabled() {
        return 6353 == this.mDelegate.getVendorId() && AUDIO_PID.contains(this.mDelegate.getProductId());
    }

    @VisibleForTesting
    Instant now() {
        return Instant.now();
    }

    public void sleep(@Nonnull Duration duration) {
        Uninterruptibles.sleepUninterruptibly(duration.toNanos(), TimeUnit.NANOSECONDS);
    }

    public void click(@Nonnull Point point) {
        this.click(point, Duration.ZERO);
    }

    private void click(Point point, Duration duration) {
        this.touch((byte)3, point, duration);
        this.touch((byte)0, point, ACTION_DELAY);
    }

    public void longClick(@Nonnull Point point) {
        this.click(point, LONG_CLICK);
    }

    public void swipe(@Nonnull Point from, @Nonnull Point to, @Nonnull Duration duration) {
        Duration elapsed;
        Instant start = this.now();
        this.touch((byte)3, from, STEP_DELAY);
        while (duration.compareTo(elapsed = Duration.between(start, this.now())) >= 0) {
            double progress = (double)elapsed.toMillis() / (double)duration.toMillis();
            Point point = new Point((int)(progress * (double)to.x + (1.0 - progress) * (double)from.x), (int)(progress * (double)to.y + (1.0 - progress) * (double)from.y));
            this.touch((byte)3, point, STEP_DELAY);
        }
        this.touch((byte)0, to, ACTION_DELAY);
    }

    private void touch(byte type, Point point, Duration pause) {
        int x = Math.min(Math.max(point.x, 0), 360);
        int y = Math.min(Math.max(point.y, 0), 640);
        byte[] data = new byte[]{type, (byte)x, (byte)(x >> 8), (byte)y, (byte)(y >> 8)};
        this.send(AoaHID.TOUCH_SCREEN, data, pause);
    }

    public void pressKeys(AoaKey ... keys) {
        this.pressKeys(Arrays.asList(keys));
    }

    public void pressKeys(@Nonnull List<AoaKey> keys) {
        Iterator it = keys.stream().filter(Objects::nonNull).iterator();
        while (it.hasNext()) {
            AoaKey key = (AoaKey)it.next();
            this.send(AoaHID.KEYBOARD, key.toHidData(), STEP_DELAY);
            this.send(AoaHID.KEYBOARD, AoaKey.NOOP.toHidData(), it.hasNext() ? STEP_DELAY : ACTION_DELAY);
        }
    }

    public void wakeUp() {
        this.send(AoaHID.SYSTEM, new byte[]{1}, ACTION_DELAY);
    }

    public void goHome() {
        this.send(AoaHID.SYSTEM, new byte[]{2}, ACTION_DELAY);
    }

    public void goBack() {
        this.send(AoaHID.SYSTEM, new byte[]{4}, ACTION_DELAY);
    }

    private void send(AoaHID hid, byte[] data, Duration pause) {
        int result = this.transfer((byte)57, hid.getId(), 0, data);
        if (result == -4) {
            LOGGER.warning(String.format("Device %s not found while sending AOA HID event; resetting connection", this.mSerialNumber));
            this.resetConnection();
            result = this.transfer((byte)57, hid.getId(), 0, data);
        }
        this.mHelper.checkResult(result);
        this.sleep(pause);
    }

    @Override
    public void close() {
        if (this.isValid()) {
            if (this.isAccessoryMode()) {
                this.unregisterHIDs();
            }
            this.mDelegate.close();
            this.mDelegate = null;
        }
    }

    public String toString() {
        return String.format("AoaDevice{%s}", this.mSerialNumber);
    }
}

