/*
 * Decompiled with CFR 0.152.
 */
package com.android.tradefed.targetprep;

import com.android.tradefed.device.DeviceNotAvailableException;
import com.android.tradefed.device.ITestDevice;
import com.android.tradefed.error.HarnessException;
import com.android.tradefed.log.LogUtil;
import com.android.tradefed.result.error.DeviceErrorIdentifier;
import com.android.tradefed.result.error.ErrorIdentifier;
import com.android.tradefed.targetprep.TargetSetupError;
import com.android.tradefed.util.AaptParser;
import com.android.tradefed.util.CommandResult;
import com.android.tradefed.util.CommandStatus;
import com.android.tradefed.util.RunUtil;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ModulePusher {
    private static final String APEX_SUFFIX = ".apex";
    private static final String APK_SUFFIX = ".apk";
    private static final String CAPEX_SUFFIX = ".capex";
    private static final String APEX_DIR = "/system/apex/";
    private static final String DISABLE_VERITY = "disable-verity";
    private static final String ENABLE_TESTHARNESS = "cmd testharness enable";
    private static final String RM_PACKAGE_CACHE = "rm -Rf /data/system/package_cache/";
    private static final String GET_APEX_PACKAGE_VERSIONS = "cmd package list packages --apex-only --show-versioncode| grep 'com.google'";
    private static final String GET_APK_PACKAGE_VERSIONS = "cmd package list packages --show-versioncode| grep 'com.google'";
    private static final String REMOUNT_COMMAND = "remount";
    private static final String GET_GOOGLE_MODULES = "pm get-moduleinfo | grep 'com.google'";
    private static final String LS_SYSTEM_APEX = "ls /system/apex/";
    private static final String PACKAGE_HEAD = "package:";
    private static final Pattern VERSION_CODE_PATTERN = Pattern.compile("package:(?<package>.+) versionCode:(?<versionCode>.+)");
    public static final String LINE_BREAK = "\\r?\\n";
    private final long mWaitTimeMs;
    private final long mDelayWaitingTimeMs;
    private final ITestDevice mDevice;
    private ImmutableMap<String, Path> mApexPathsUnderSystem;

    public ModulePusher(ITestDevice device, long waitTimeMs, long delayWaitingTimeMs) {
        this.mDevice = device;
        this.mWaitTimeMs = waitTimeMs;
        this.mDelayWaitingTimeMs = delayWaitingTimeMs;
    }

    public void installModules(ImmutableMultimap<String, File> moduleFiles, boolean factoryReset, boolean disablePackageCache) throws TargetSetupError, DeviceNotAvailableException, ModulePushError {
        ITestDevice device = this.mDevice;
        this.setupDevice(device);
        ModulePusher.checkPreloadModules(device);
        List<ModuleInfo> pushedModules = this.pushModulesToDevice(device, moduleFiles);
        if (disablePackageCache) {
            this.cleanPackageCache(device);
        }
        this.reloadAllModules(device, factoryReset);
        this.waitForDeviceToBeResponsive(this.mWaitTimeMs);
        this.checkApexActivated(device, pushedModules);
        LogUtil.CLog.i("Check pushed module version code after device reboot");
        this.checkModuleVersions(device, pushedModules);
    }

    @VisibleForTesting
    protected void setupDevice(ITestDevice device) throws DeviceNotAvailableException, ModulePushError {
        device.enableAdbRoot();
        String disableVerity = device.executeAdbCommand(DISABLE_VERITY);
        LogUtil.CLog.i("disable-verity status: %s", disableVerity);
        if (!disableVerity.contains("disabled")) {
            throw new ModulePushError(String.format("Failed to disable verity on device %s", device.getSerialNumber()), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_FAILED_TO_RESET);
        }
        LogUtil.CLog.d("disable-verity status: %s", disableVerity);
        device.reboot();
        ModulePusher.remountDevice(device);
        device.reboot();
        ModulePusher.remountDevice(device);
    }

    static void remountDevice(ITestDevice device) throws ModulePushError {
        String remount;
        try {
            device.enableAdbRoot();
            remount = device.executeAdbCommand(REMOUNT_COMMAND);
        }
        catch (DeviceNotAvailableException e) {
            throw new ModulePushError(String.format("Failed to execute remount command to %s", device.getSerialNumber()), e, DeviceErrorIdentifier.DEVICE_FAILED_TO_REMOUNT);
        }
        LogUtil.CLog.i("adb remount status: %s", remount);
        if (!remount.contains("remount succeed")) {
            throw new ModulePushError(String.format("Failed to remount device on %s. Error log: '%s'", device.getSerialNumber(), remount), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_FAILED_TO_REMOUNT);
        }
        LogUtil.CLog.i("Remount Success, output is %s", remount);
    }

    static void checkPreloadModules(ITestDevice device) throws DeviceNotAvailableException, ModulePushError {
        Set<ITestDevice.ApexInfo> activatedApexes = device.getActiveApexes();
        LogUtil.CLog.i("Activated apex packages list before module push:");
        for (ITestDevice.ApexInfo info : activatedApexes) {
            LogUtil.CLog.i("Activated apex: %s", info.toString());
        }
        LogUtil.CLog.i("Preloaded modules:");
        String out = device.executeShellV2Command(GET_GOOGLE_MODULES).getStdout();
        if (out == null) {
            throw new ModulePushError("no modules preloaded", (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
        }
        LogUtil.CLog.i("Preload modules are as below: \n%s", out);
    }

    List<ModuleInfo> pushModulesToDevice(ITestDevice device, ImmutableMultimap<String, File> moduleFiles) throws DeviceNotAvailableException, ModulePushError {
        ArrayList<ModuleInfo> pushedModules = new ArrayList<ModuleInfo>();
        int apiLevel = device.getApiLevel();
        ImmutableMap<String, String> versionCodes = this.getGooglePackageVersionCodesOnDevice(device);
        for (String packageName : moduleFiles.keySet()) {
            if (versionCodes.containsKey(packageName)) continue;
            throw new ModulePushError(String.format("Can't get version code of %s on the device %s.", packageName, device.getSerialNumber()), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
        }
        for (String packageName : moduleFiles.keySet()) {
            File[] toPush = ((ImmutableCollection)moduleFiles.get((Object)packageName)).toArray(new File[0]);
            pushedModules.add(this.pushFile(toPush, device, packageName, apiLevel, versionCodes.get(packageName)));
        }
        return pushedModules;
    }

    ModuleInfo pushFile(File[] moduleFiles, ITestDevice device, String packageName, int apiLevel, String preloadVersion) throws DeviceNotAvailableException, ModulePushError {
        Path toPush;
        Path toRename;
        boolean isDir;
        if (moduleFiles.length == 0) {
            throw new ModulePushError("No file to push.", (ErrorIdentifier)DeviceErrorIdentifier.FAIL_PUSH_FILE);
        }
        ModuleInfo moduleInfo = this.retrieveModuleInfo(moduleFiles[0]);
        LogUtil.CLog.i("To update module %s from version %s to %s", packageName, preloadVersion, moduleInfo.versionCode());
        Path[] preloadPaths = this.getPreloadPaths(device, moduleFiles, packageName, apiLevel);
        Path packagePath = preloadPaths[0];
        boolean bl = isDir = preloadPaths.length > 1;
        if (isDir) {
            device.deleteFile(packagePath.toString());
            toRename = moduleFiles[0].toPath().getParent();
        } else {
            toRename = moduleFiles[0].toPath();
        }
        Path target = toRename.getParent().resolve(packagePath.getFileName());
        try {
            toPush = Files.move(toRename, target, new CopyOption[0]);
            LogUtil.CLog.i("Local file name %s changed to the preload name %s", toRename, target);
        }
        catch (IOException e) {
            throw new ModulePushError(String.format("Failed to rename File '%s' to the name of '%s'", toRename, target), e, DeviceErrorIdentifier.FAIL_PUSH_FILE);
        }
        this.pushPackageToDevice(device, toPush.toFile(), packagePath.toString(), isDir);
        this.waitForDeviceToBeResponsive(this.mDelayWaitingTimeMs);
        return moduleInfo;
    }

    private ImmutableMap<String, String> getGooglePackageVersionCodesOnDevice(ITestDevice device) throws DeviceNotAvailableException {
        ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
        String outputs = device.executeShellV2Command(GET_APEX_PACKAGE_VERSIONS).getStdout();
        this.waitForDeviceToBeResponsive(this.mDelayWaitingTimeMs);
        LogUtil.CLog.i("Apex version code output string is:\n%s", outputs);
        builder.putAll(this.parsePackageVersionCodes(outputs));
        outputs = device.executeShellV2Command(GET_APK_PACKAGE_VERSIONS).getStdout();
        this.waitForDeviceToBeResponsive(this.mDelayWaitingTimeMs);
        LogUtil.CLog.i("Apk version code output string is:\n%s", outputs);
        builder.putAll(this.parsePackageVersionCodes(outputs));
        return builder.build();
    }

    @VisibleForTesting
    protected ImmutableMap<String, String> parsePackageVersionCodes(String output) {
        ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
        for (String line : output.split(LINE_BREAK)) {
            Matcher matcher = VERSION_CODE_PATTERN.matcher(line.trim());
            if (matcher.matches()) {
                builder.put(matcher.group("package"), matcher.group("versionCode"));
                continue;
            }
            LogUtil.CLog.w("The line %s doesn't match the pattern of version code. Skip the line.", line);
        }
        return builder.build();
    }

    @VisibleForTesting
    protected Path[] getPreloadPaths(ITestDevice device, File[] moduleFiles, String packageName, int apiLevel) throws DeviceNotAvailableException, ModulePushError {
        Object[] paths;
        try {
            paths = this.getPathsOnDevice(device, packageName);
        }
        catch (ModulePushError e) {
            if (apiLevel < 31 && ModulePusher.hasExtension(APEX_SUFFIX, moduleFiles[0])) {
                return new Path[]{Paths.get(APEX_DIR, packageName + APEX_SUFFIX)};
            }
            LogUtil.CLog.w("Failed to get path of package %s on API level %d", packageName, apiLevel);
            throw e;
        }
        if (paths.length > 1) {
            ArrayList<Path> res = new ArrayList<Path>();
            Path parentDir = Paths.get(paths[0], new String[0]).getParent();
            if (!this.isPackageDir(device, parentDir, paths.length)) {
                throw new ModulePushError(String.format("The parent folder %s contains files not related to the package %s", parentDir, packageName), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
            }
            res.add(parentDir);
            for (Object filePath : paths) {
                res.add(Paths.get((String)filePath, new String[0]));
            }
            return res.toArray(new Path[0]);
        }
        if (ModulePusher.hasExtension(APEX_SUFFIX, moduleFiles[0]) && !paths[0].startsWith(APEX_DIR)) {
            LogUtil.CLog.w("The path of the system apex is not /system/apex. Actual source paths are: %s. Expect to override them with packages in %s after reboot.", Arrays.toString(paths), APEX_DIR);
            return new Path[]{this.getApexPathUnderSystem(device, packageName)};
        }
        return new Path[]{Paths.get(paths[0], new String[0])};
    }

    boolean isPackageDir(ITestDevice device, Path dirPathOnDevice, int packageFileNum) throws DeviceNotAvailableException, ModulePushError {
        CommandResult lsResult = device.executeShellV2Command(String.format("ls %s", dirPathOnDevice));
        if (!CommandStatus.SUCCESS.equals((Object)lsResult.getStatus())) {
            throw new ModulePushError(String.format("Failed to ls files in %s", dirPathOnDevice), (ErrorIdentifier)DeviceErrorIdentifier.SHELL_COMMAND_ERROR);
        }
        return packageFileNum == lsResult.getStdout().split(LINE_BREAK).length;
    }

    @VisibleForTesting
    protected String[] getPathsOnDevice(ITestDevice device, String packageName) throws DeviceNotAvailableException, ModulePushError {
        String output = device.executeShellV2Command("pm path " + packageName).getStdout();
        String[] lines = output.split(LINE_BREAK);
        if (!output.contains(PACKAGE_HEAD) || lines.length == 0) {
            throw new ModulePushError(String.format("Failed to find file paths of %s on the device %s. Error log\n '%s'", packageName, device.getSerialNumber(), output), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
        }
        ArrayList<String> paths = new ArrayList<String>();
        for (String line : lines) {
            paths.add(line.substring(PACKAGE_HEAD.length()).trim());
        }
        return paths.toArray(new String[0]);
    }

    @VisibleForTesting
    protected Path getApexPathUnderSystem(ITestDevice device, String packageName) throws ModulePushError, DeviceNotAvailableException {
        ImmutableMap<String, Path> apexPaths = this.getApexPaths(device);
        if (apexPaths.containsKey(packageName)) {
            return (Path)apexPaths.get(packageName);
        }
        throw new ModulePushError(String.format("No apex under /system/apex available for %s. Print ls results %s", packageName, apexPaths), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
    }

    private ImmutableMap<String, Path> getApexPaths(ITestDevice device) throws DeviceNotAvailableException {
        if (this.mApexPathsUnderSystem == null) {
            ImmutableMap.Builder<String, Path> builder = ImmutableMap.builder();
            String outputs = device.executeShellV2Command(LS_SYSTEM_APEX).getStdout();
            LogUtil.CLog.i("ls /system/apex/ output string is:\n%s", outputs);
            for (String line : outputs.split(LINE_BREAK)) {
                String fileName = line.trim();
                int endIndex = -1;
                if (fileName.contains("_")) {
                    endIndex = fileName.indexOf(95);
                } else if (fileName.contains(APEX_SUFFIX)) {
                    endIndex = fileName.indexOf(APEX_SUFFIX);
                } else if (fileName.contains(CAPEX_SUFFIX)) {
                    endIndex = fileName.indexOf(CAPEX_SUFFIX);
                }
                if (endIndex > 0) {
                    builder.put(fileName.substring(0, endIndex), Paths.get(APEX_DIR, fileName));
                    continue;
                }
                LogUtil.CLog.w("Got unexpected filename %s under /system/apex/", fileName);
            }
            this.mApexPathsUnderSystem = builder.build();
        }
        return this.mApexPathsUnderSystem;
    }

    void pushPackageToDevice(ITestDevice device, File localFile, String filePathOnDevice, boolean isDir) throws DeviceNotAvailableException, ModulePushError {
        boolean success;
        boolean bl = success = isDir ? device.pushDir(localFile, filePathOnDevice) : device.pushFile(localFile, filePathOnDevice);
        if (!success) {
            throw new ModulePushError(String.format("Failed to push File '%s' to '%s' on device %s.", localFile, filePathOnDevice, device.getSerialNumber()), (ErrorIdentifier)DeviceErrorIdentifier.FAIL_PUSH_FILE);
        }
        LogUtil.CLog.i("Local file %s got pushed to the preload path '%s", localFile.getName(), filePathOnDevice);
    }

    @VisibleForTesting
    protected void waitForDeviceToBeResponsive(long waitTime) {
        RunUtil.getDefault().sleep(waitTime);
    }

    void cleanPackageCache(ITestDevice device) throws DeviceNotAvailableException {
        device.executeShellV2Command(RM_PACKAGE_CACHE);
    }

    void reloadAllModules(ITestDevice device, boolean factory_reset) throws DeviceNotAvailableException, TargetSetupError {
        if (factory_reset) {
            CommandResult cr = device.executeShellV2Command(ENABLE_TESTHARNESS);
            if (!CommandStatus.SUCCESS.equals((Object)cr.getStatus())) {
                LogUtil.CLog.e("Failed to enable test harness mode: %s", cr.toString());
                throw new TargetSetupError(cr.toString(), device.getDeviceDescriptor());
            }
        } else {
            device.reboot();
        }
        device.waitForDeviceAvailable(this.mWaitTimeMs);
    }

    protected void checkApexActivated(ITestDevice device, List<ModuleInfo> modules) throws DeviceNotAvailableException, ModulePushError {
        Set<ITestDevice.ApexInfo> activatedApexes = device.getActiveApexes();
        if (activatedApexes.isEmpty()) {
            throw new ModulePushError(String.format("Failed to retrieve activated apex on device %s. Empty set returned.", device.getSerialNumber()), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
        }
        LogUtil.CLog.i("Activated apex packages list after module/train installation:");
        for (ITestDevice.ApexInfo info : activatedApexes) {
            LogUtil.CLog.i("Activated apex: %s", info.toString());
        }
        List<ModuleInfo> failToActivateApex = this.getModulesFailToActivate(modules, activatedApexes);
        if (!failToActivateApex.isEmpty()) {
            throw new ModulePushError(String.format("Failed to activate %s on device %s.", failToActivateApex, device.getSerialNumber()), (ErrorIdentifier)DeviceErrorIdentifier.FAIL_ACTIVATE_APEX);
        }
        LogUtil.CLog.i("Train activation succeed.");
    }

    protected List<ModuleInfo> getModulesFailToActivate(List<ModuleInfo> toInstall, Set<ITestDevice.ApexInfo> activatedApexes) {
        ArrayList<ModuleInfo> failToActivateApex = new ArrayList<ModuleInfo>();
        HashMap<String, ITestDevice.ApexInfo> activatedApexInfo = new HashMap<String, ITestDevice.ApexInfo>();
        for (ITestDevice.ApexInfo info : activatedApexes) {
            activatedApexInfo.put(info.name, info);
        }
        for (ModuleInfo moduleInfo : toInstall) {
            if (moduleInfo.isApk()) continue;
            if (!activatedApexInfo.containsKey(moduleInfo.packageName())) {
                failToActivateApex.add(moduleInfo);
                continue;
            }
            if (((ITestDevice.ApexInfo)activatedApexInfo.get((Object)moduleInfo.packageName())).versionCode == Long.parseLong(moduleInfo.versionCode())) continue;
            failToActivateApex.add(moduleInfo);
        }
        return failToActivateApex;
    }

    void checkModuleVersions(ITestDevice device, List<ModuleInfo> pushedModules) throws DeviceNotAvailableException, ModulePushError {
        ImmutableMap<String, String> newVersionCodes = this.getGooglePackageVersionCodesOnDevice(device);
        for (ModuleInfo moduleInfo : pushedModules) {
            if (!newVersionCodes.containsKey(moduleInfo.packageName()) || !moduleInfo.versionCode().equals(newVersionCodes.get(moduleInfo.packageName()))) {
                throw new ModulePushError("Failed to install package " + moduleInfo.packageName(), (ErrorIdentifier)DeviceErrorIdentifier.DEVICE_UNEXPECTED_RESPONSE);
            }
            LogUtil.CLog.i("Packages %s pushed! The expected version code is: %s, the actual is: %s", moduleInfo.packageName(), moduleInfo.versionCode(), newVersionCodes.get(moduleInfo.packageName()));
        }
    }

    static boolean hasExtension(String extension, File fileName) {
        return fileName.getName().endsWith(extension);
    }

    @VisibleForTesting
    ModuleInfo retrieveModuleInfo(File packageFile) throws ModulePushError {
        AaptParser parser = AaptParser.parse(packageFile);
        if (parser == null) {
            throw new ModulePushError(String.format("Failed to parse package file %s", packageFile), (ErrorIdentifier)DeviceErrorIdentifier.AAPT_PARSER_FAILED);
        }
        return ModuleInfo.create(parser.getPackageName(), parser.getVersionCode(), packageFile.getName().endsWith(APK_SUFFIX));
    }

    public static class ModulePushError
    extends HarnessException {
        public ModulePushError(String message2, Throwable cause, ErrorIdentifier errorId) {
            super(message2, cause, errorId);
        }

        public ModulePushError(String message2, ErrorIdentifier errorId) {
            super(message2, errorId);
        }
    }

    static final class ModuleInfo {
        private final String packageName;
        private final String versionCode;
        private final boolean isApk;

        public String packageName() {
            return this.packageName;
        }

        public String versionCode() {
            return this.versionCode;
        }

        public boolean isApk() {
            return this.isApk;
        }

        ModuleInfo(String packageName, String versionCode, boolean isApk) {
            this.packageName = packageName;
            this.versionCode = versionCode;
            this.isApk = isApk;
        }

        public static ModuleInfo create(String packageName, String versionCode, boolean isApk) {
            return new ModuleInfo(packageName, versionCode, isApk);
        }

        public boolean equals(Object other) {
            if (other instanceof ModuleInfo) {
                ModuleInfo moduleInfo = (ModuleInfo)other;
                return this.packageName.equals(moduleInfo.packageName) && this.versionCode.equals(moduleInfo.versionCode) && this.isApk == moduleInfo.isApk;
            }
            return false;
        }

        public int hashCode() {
            return this.toString().hashCode();
        }

        public String toString() {
            return String.format("{packageName: %s, versionCode: %s, isApk: %b}", this.packageName, this.versionCode, this.isApk);
        }
    }
}

