/*
 * Decompiled with CFR 0.152.
 */
package com.android.tradefed.util;

import com.android.tradefed.device.DeviceNotAvailableException;
import com.android.tradefed.device.ITestDevice;
import com.android.tradefed.log.LogUtil;
import com.android.tradefed.util.sl4a.Sl4aClient;
import com.android.tradefed.util.sl4a.Sl4aEventDispatcher;
import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class Sl4aBluetoothUtil {
    private static final long BT_STATE_CHANGE_TIMEOUT_MS = 10000L;
    private static final long BT_PAIRING_CHECK_INTERVAL_MS = 200L;
    private static final long BT_CHECK_CONNECTION_INTERVAL_MS = 100L;
    @VisibleForTesting
    static final String BT_SNOOP_LOG_CMD_LEGACY = "setprop persist.bluetooth.btsnoopenable %s";
    @VisibleForTesting
    static final String BT_SNOOP_LOG_CMD = "setprop persist.bluetooth.btsnooplogmode %s";
    private Map<String, Sl4aClient> mSl4aClients = new HashMap<String, Sl4aClient>();
    private Map<ITestDevice, String> mAddresses = new HashMap<ITestDevice, String>();
    private Duration mBtPairTimeout = Duration.ofSeconds(25L);
    private Duration mBtConnectionTimeout = Duration.ofSeconds(15L);

    public void setBtPairTimeout(Duration timeout) {
        this.mBtPairTimeout = timeout;
    }

    public void setBtConnectionTimeout(Duration timeout) {
        this.mBtConnectionTimeout = timeout;
    }

    public void startSl4a(ITestDevice device, File sl4aApkFile) throws DeviceNotAvailableException {
        Sl4aClient sl4aClient = Sl4aClient.startSL4A(device, sl4aApkFile);
        this.mSl4aClients.put(device.getSerialNumber(), sl4aClient);
    }

    public void stopSl4a() {
        for (Map.Entry<String, Sl4aClient> entry : this.mSl4aClients.entrySet()) {
            entry.getValue().close();
        }
        this.mSl4aClients.clear();
    }

    @VisibleForTesting
    void setSl4a(ITestDevice device, Sl4aClient client) {
        this.mSl4aClients.put(device.getSerialNumber(), client);
    }

    protected void finalize() {
        this.stopSl4a();
    }

    public boolean enable(ITestDevice device) throws DeviceNotAvailableException {
        return this.toggleState(device, true);
    }

    public boolean disable(ITestDevice device) throws DeviceNotAvailableException {
        return this.toggleState(device, false);
    }

    public String getAddress(ITestDevice device) throws DeviceNotAvailableException {
        if (this.mAddresses.containsKey(device)) {
            return this.mAddresses.get(device);
        }
        Sl4aClient client = this.getSl4aClient(device);
        String address = null;
        try {
            address = (String)client.rpcCall("bluetoothGetLocalAddress", new Object[0]);
            this.mAddresses.put(device, address);
        }
        catch (IOException e) {
            LogUtil.CLog.e("Failed to get Bluetooth MAC address on device: %s, %s", device.getSerialNumber(), e);
        }
        return address;
    }

    public Set<String> getBondedDevices(ITestDevice device) throws DeviceNotAvailableException {
        HashSet<String> addresses = new HashSet<String>();
        Sl4aClient client = this.getSl4aClient(device);
        try {
            Object response = client.rpcCall("bluetoothGetBondedDevices", new Object[0]);
            if (response != null) {
                JSONArray bondedDevices = (JSONArray)response;
                for (int i = 0; i < bondedDevices.length(); ++i) {
                    JSONObject bondedDevice = bondedDevices.getJSONObject(i);
                    if (!bondedDevice.has("address")) continue;
                    addresses.add(bondedDevice.getString("address"));
                }
            }
        }
        catch (IOException | JSONException e) {
            LogUtil.CLog.e("Failed to get bonded devices for device: %s, %s", device.getSerialNumber(), e);
        }
        return addresses;
    }

    public boolean pair(ITestDevice primary, ITestDevice secondary) throws DeviceNotAvailableException {
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        Sl4aClient secondaryClient = this.getSl4aClient(secondary);
        try {
            if (this.isPaired(primary, secondary)) {
                LogUtil.CLog.i("The two devices are already paired.");
                return true;
            }
            LogUtil.CLog.d("Make secondary device discoverable");
            secondaryClient.rpcCall("bluetoothMakeDiscoverable", new Object[0]);
            Integer response = (Integer)secondaryClient.rpcCall("bluetoothGetScanMode", new Object[0]);
            if (response != 3) {
                LogUtil.CLog.e("Scan mode is not CONNECTABLE_DISCOVERABLE");
                return false;
            }
            LogUtil.CLog.d("Secondary device is made discoverable");
            LogUtil.CLog.d("Start pairing helper on both devices");
            primaryClient.rpcCall("bluetoothStartPairingHelper", new Object[0]);
            secondaryClient.rpcCall("bluetoothStartPairingHelper", new Object[0]);
            LogUtil.CLog.d("Start discover and bond to secondary device: %s", secondary.getSerialNumber());
            primaryClient.getEventDispatcher().clearAllEvents();
            primaryClient.rpcCall("bluetoothDiscoverAndBond", this.getAddress(secondary));
            if (!this.waitUntilPaired(primary, secondary)) {
                LogUtil.CLog.e("Bluetooth pairing timeout");
                return false;
            }
        }
        catch (IOException | InterruptedException e) {
            LogUtil.CLog.e("Error when pair two devices, %s", e);
            return false;
        }
        LogUtil.CLog.i("Secondary device successfully paired");
        return true;
    }

    public boolean unpairAll(ITestDevice device) throws DeviceNotAvailableException {
        Set<String> bondedDevices = this.getBondedDevices(device);
        Sl4aClient client = this.getSl4aClient(device);
        for (String address : bondedDevices) {
            try {
                Boolean res = (Boolean)client.rpcCall("bluetoothUnbond", address);
                if (res.booleanValue()) continue;
                LogUtil.CLog.w("Failed to unpair device %s. It may not be an actual failure, instead it may be due to trying to unpair an already unpaired device. This usually happens when device was first connected using LE transport where both LE address and classic address are paired and unpaired at the same time.", address);
            }
            catch (IOException e) {
                LogUtil.CLog.e("Failed to unpair all Bluetooth devices, %s", e);
            }
        }
        return this.getBondedDevices(device).isEmpty();
    }

    public boolean connect(ITestDevice primary, ITestDevice secondary, Set<BluetoothProfile> profiles) throws DeviceNotAvailableException {
        if (!this.isPaired(primary, secondary)) {
            LogUtil.CLog.e("Primary device have not yet paired to secondary device");
            return false;
        }
        LogUtil.CLog.d("Connecting to profiles: %s", profiles);
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        String address = this.getAddress(secondary);
        try {
            primaryClient.rpcCall("bluetoothStartConnectionStateChangeMonitor", address);
            primaryClient.rpcCall("bluetoothConnectBonded", address);
            Set<BluetoothProfile> connectedProfiles = this.waitForConnectedOrDisconnectedProfiles(primary, address, BluetoothConnectionState.CONNECTED, profiles);
            return this.waitForRemainingProfilesConnected(primary, address, connectedProfiles, profiles);
        }
        catch (IOException | InterruptedException | JSONException e) {
            LogUtil.CLog.e("Failed to connect to secondary device, %s", e);
            return false;
        }
    }

    public boolean disconnect(ITestDevice primary, ITestDevice secondary, Set<BluetoothProfile> profiles) throws DeviceNotAvailableException {
        LogUtil.CLog.d("Disconnecting to profiles: %s", profiles);
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        String address = this.getAddress(secondary);
        try {
            primaryClient.rpcCall("bluetoothStartConnectionStateChangeMonitor", address);
            primaryClient.rpcCall("bluetoothDisconnectConnectedProfile", address, new JSONArray(profiles.stream().map(profile -> profile.getProfile()).collect(Collectors.toList())));
            Set<BluetoothProfile> disconnectedProfiles = this.waitForConnectedOrDisconnectedProfiles(primary, address, BluetoothConnectionState.DISCONNECTED, profiles);
            return this.waitForRemainingProfilesDisconnected(primary, address, disconnectedProfiles, profiles);
        }
        catch (IOException | InterruptedException | JSONException e) {
            LogUtil.CLog.e("Failed to disconnect from secondary device, %s", e);
            return false;
        }
    }

    public boolean enableBluetoothSnoopLog(ITestDevice device) throws DeviceNotAvailableException {
        if (this.isQAndAbove(device)) {
            device.executeShellCommand(String.format(BT_SNOOP_LOG_CMD, "full"));
        } else {
            device.executeShellCommand(String.format(BT_SNOOP_LOG_CMD_LEGACY, "true"));
        }
        return this.disable(device) && this.enable(device);
    }

    public boolean disableBluetoothSnoopLog(ITestDevice device) throws DeviceNotAvailableException {
        if (this.isQAndAbove(device)) {
            device.executeShellCommand(String.format(BT_SNOOP_LOG_CMD, "disabled"));
        } else {
            device.executeShellCommand(String.format(BT_SNOOP_LOG_CMD_LEGACY, "false"));
        }
        return this.disable(device) && this.enable(device);
    }

    public boolean changeProfileAccessPermission(ITestDevice primary, ITestDevice secondary, BluetoothProfile profile, BluetoothAccessLevel access) throws DeviceNotAvailableException {
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        String secondaryAddress = this.getAddress(secondary);
        try {
            primaryClient.rpcCall("bluetoothChangeProfileAccessPermission", secondaryAddress, profile.mProfile, access.getAccess());
        }
        catch (IOException e) {
            LogUtil.CLog.e("Failed to set profile access level %s for profile %s, %s", new Object[]{access, profile, e});
            return false;
        }
        return true;
    }

    public boolean setProfilePriority(ITestDevice primary, ITestDevice secondary, Set<BluetoothProfile> profiles, BluetoothPriorityLevel priority) throws DeviceNotAvailableException {
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        String secondaryAddress = this.getAddress(secondary);
        for (BluetoothProfile profile : profiles) {
            try {
                switch (profile) {
                    case A2DP_SINK: {
                        primaryClient.rpcCall("bluetoothA2dpSinkSetPriority", secondaryAddress, priority.getPriority());
                        break;
                    }
                    case HEADSET_CLIENT: {
                        primaryClient.rpcCall("bluetoothHfpClientSetPriority", secondaryAddress, priority.getPriority());
                        break;
                    }
                    case PBAP_CLIENT: {
                        primaryClient.rpcCall("bluetoothPbapClientSetPriority", secondaryAddress, priority.getPriority());
                        break;
                    }
                    default: {
                        LogUtil.CLog.e("Profile %s is not yet supported for priority settings", new Object[]{profile});
                        return false;
                    }
                }
            }
            catch (IOException e) {
                LogUtil.CLog.e("Failed to set profile %s with priority %s, %s", new Object[]{profile, priority, e});
                return false;
            }
        }
        return true;
    }

    private boolean toggleState(ITestDevice device, boolean targetState) throws DeviceNotAvailableException {
        Sl4aClient client = this.getSl4aClient(device);
        try {
            boolean currentState = (Boolean)client.rpcCall("bluetoothCheckState", new Object[0]);
            if (currentState == targetState) {
                return true;
            }
            client.getEventDispatcher().clearAllEvents();
            Boolean result = (Boolean)client.rpcCall("bluetoothToggleState", targetState);
            if (!result.booleanValue()) {
                LogUtil.CLog.e("Error in sl4a when toggling %s Bluetooth state.", targetState ? "ON" : "OFF");
                return false;
            }
            String event = targetState ? "BluetoothStateChangedOn" : "BluetoothStateChangedOff";
            Sl4aEventDispatcher.EventSl4aObject response = client.getEventDispatcher().popEvent(event, 10000L);
            if (response == null) {
                LogUtil.CLog.e("Get null response after toggling %s Bluetooth state.", targetState ? "ON" : "OFF");
                return false;
            }
        }
        catch (IOException e) {
            LogUtil.CLog.e("Error when toggling %s Bluetooth state, %s", targetState ? "ON" : "OFF", e);
            return false;
        }
        return true;
    }

    private Sl4aClient getSl4aClient(ITestDevice device) throws DeviceNotAvailableException {
        String serial = device.getSerialNumber();
        if (!this.mSl4aClients.containsKey(serial)) {
            Sl4aClient client = Sl4aClient.startSL4A(device, null);
            this.mSl4aClients.put(serial, client);
        }
        return this.mSl4aClients.get(serial);
    }

    private boolean waitUntilPaired(ITestDevice primary, ITestDevice secondary) throws InterruptedException, DeviceNotAvailableException {
        long endTime = System.currentTimeMillis() + this.mBtPairTimeout.toMillis();
        while (System.currentTimeMillis() < endTime) {
            if (this.isPaired(primary, secondary)) {
                return true;
            }
            Thread.sleep(200L);
        }
        return false;
    }

    private boolean isPaired(ITestDevice primary, ITestDevice secondary) throws DeviceNotAvailableException {
        return this.getBondedDevices(primary).contains(this.getAddress(secondary));
    }

    private Set<BluetoothProfile> waitForConnectedOrDisconnectedProfiles(ITestDevice primary, String address, BluetoothConnectionState targetState, Set<BluetoothProfile> allProfiles) throws DeviceNotAvailableException, JSONException {
        HashSet<BluetoothProfile> targetProfiles = new HashSet<BluetoothProfile>();
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        while (!targetProfiles.containsAll(allProfiles)) {
            Sl4aEventDispatcher.EventSl4aObject event = primaryClient.getEventDispatcher().popEvent("BluetoothProfileConnectionStateChanged", this.mBtConnectionTimeout.toMillis());
            if (event == null) {
                LogUtil.CLog.w("Timeout while waiting for connection state changes for all profiles");
                return targetProfiles;
            }
            JSONObject profileData = new JSONObject(event.getData());
            int profile = profileData.getInt("profile");
            int state = profileData.getInt("state");
            String actualAddress = profileData.getString("addr");
            if (state != targetState.getState() || !address.equals(actualAddress)) continue;
            targetProfiles.add(BluetoothProfile.valueOfProfile(profile));
        }
        return targetProfiles;
    }

    private boolean waitForRemainingProfilesConnected(ITestDevice primary, String address, Set<BluetoothProfile> connectedProfiles, Set<BluetoothProfile> allProfiles) throws InterruptedException, DeviceNotAvailableException, JSONException, IOException {
        long endTime = System.currentTimeMillis() + this.mBtConnectionTimeout.toMillis();
        while (System.currentTimeMillis() < endTime && !connectedProfiles.containsAll(allProfiles)) {
            for (BluetoothProfile profile : allProfiles) {
                if (connectedProfiles.contains((Object)profile)) continue;
                if (this.isProfileConnected(primary, address, profile)) {
                    connectedProfiles.add(profile);
                }
                LogUtil.CLog.d("Connected profiles for now: %s", connectedProfiles);
            }
            Thread.sleep(100L);
        }
        return connectedProfiles.containsAll(allProfiles);
    }

    private boolean waitForRemainingProfilesDisconnected(ITestDevice primary, String address, Set<BluetoothProfile> disConnectedProfiles, Set<BluetoothProfile> allProfiles) throws InterruptedException, DeviceNotAvailableException, JSONException, IOException {
        long endTime = System.currentTimeMillis() + this.mBtConnectionTimeout.toMillis();
        while (System.currentTimeMillis() < endTime && !disConnectedProfiles.containsAll(allProfiles)) {
            for (BluetoothProfile profile : allProfiles) {
                if (disConnectedProfiles.contains((Object)profile)) continue;
                if (!this.isProfileConnected(primary, address, profile)) {
                    disConnectedProfiles.add(profile);
                }
                LogUtil.CLog.d("Disconnected profiles for now: %s", disConnectedProfiles);
            }
            Thread.sleep(100L);
        }
        return disConnectedProfiles.containsAll(allProfiles);
    }

    private boolean isProfileConnected(ITestDevice primary, String address, BluetoothProfile profile) throws DeviceNotAvailableException, IOException, JSONException {
        switch (profile) {
            case HEADSET_CLIENT: {
                return this.checkConnectedDevice(primary, address, "bluetoothHfpClientGetConnectedDevices");
            }
            case A2DP: {
                return this.checkConnectedDevice(primary, address, "bluetoothA2dpGetConnectedDevices");
            }
            case A2DP_SINK: {
                return this.checkConnectedDevice(primary, address, "bluetoothA2dpSinkGetConnectedDevices");
            }
            case PAN: {
                return this.checkConnectedDevice(primary, address, "bluetoothPanGetConnectedDevices");
            }
            case PBAP_CLIENT: {
                return this.checkConnectedDevice(primary, address, "bluetoothPbapClientGetConnectedDevices");
            }
            case MAP: {
                return this.checkConnectedDevice(primary, address, "bluetoothMapGetConnectedDevices");
            }
            case MAP_CLIENT: {
                return this.checkConnectedDevice(primary, address, "bluetoothMapClientGetConnectedDevices");
            }
        }
        LogUtil.CLog.e("Unsupported profile %s to check connection state", new Object[]{profile});
        return false;
    }

    private boolean checkConnectedDevice(ITestDevice primary, String address, String sl4aCommand) throws DeviceNotAvailableException, IOException, JSONException {
        Sl4aClient primaryClient = this.getSl4aClient(primary);
        JSONArray devices = (JSONArray)primaryClient.rpcCall(sl4aCommand, new Object[0]);
        if (devices == null) {
            LogUtil.CLog.e("Empty response");
            return false;
        }
        for (int i = 0; i < devices.length(); ++i) {
            JSONObject device = devices.getJSONObject(i);
            if (!device.has("address") || !device.getString("address").equals(address)) continue;
            return true;
        }
        return false;
    }

    private boolean isQAndAbove(ITestDevice device) throws DeviceNotAvailableException {
        return device.getApiLevel() > 28;
    }

    @VisibleForTesting
    static class Commands {
        static final String BLUETOOTH_CHECK_STATE = "bluetoothCheckState";
        static final String BLUETOOTH_TOGGLE_STATE = "bluetoothToggleState";
        static final String BLUETOOTH_GET_LOCAL_ADDRESS = "bluetoothGetLocalAddress";
        static final String BLUETOOTH_GET_BONDED_DEVICES = "bluetoothGetBondedDevices";
        static final String BLUETOOTH_MAKE_DISCOVERABLE = "bluetoothMakeDiscoverable";
        static final String BLUETOOTH_GET_SCAN_MODE = "bluetoothGetScanMode";
        static final String BLUETOOTH_START_PAIRING_HELPER = "bluetoothStartPairingHelper";
        static final String BLUETOOTH_DISCOVER_AND_BOND = "bluetoothDiscoverAndBond";
        static final String BLUETOOTH_UNBOND = "bluetoothUnbond";
        static final String BLUETOOTH_START_CONNECTION_STATE_CHANGE_MONITOR = "bluetoothStartConnectionStateChangeMonitor";
        static final String BLUETOOTH_CONNECT_BONDED = "bluetoothConnectBonded";
        static final String BLUETOOTH_DISCONNECT_CONNECTED_PROFILE = "bluetoothDisconnectConnectedProfile";
        static final String BLUETOOTH_HFP_CLIENT_GET_CONNECTED_DEVICES = "bluetoothHfpClientGetConnectedDevices";
        static final String BLUETOOTH_A2DP_GET_CONNECTED_DEVICES = "bluetoothA2dpGetConnectedDevices";
        static final String BLUETOOTH_A2DP_SINK_GET_CONNECTED_DEVICES = "bluetoothA2dpSinkGetConnectedDevices";
        static final String BLUETOOTH_PBAP_CLIENT_GET_CONNECTED_DEVICES = "bluetoothPbapClientGetConnectedDevices";
        static final String BLUETOOTH_PAN_GET_CONNECTED_DEVICES = "bluetoothPanGetConnectedDevices";
        static final String BLUETOOTH_MAP_GET_CONNECTED_DEVICES = "bluetoothMapGetConnectedDevices";
        static final String BLUETOOTH_MAP_CLIENT_GET_CONNECTED_DEVICES = "bluetoothMapClientGetConnectedDevices";
        static final String BLUETOOTH_CHANGE_PROFILE_ACCESS_PERMISSION = "bluetoothChangeProfileAccessPermission";
        static final String BLUETOOTH_A2DP_SINK_SET_PRIORITY = "bluetoothA2dpSinkSetPriority";
        static final String BLUETOOTH_HFP_CLIENT_SET_PRIORITY = "bluetoothHfpClientSetPriority";
        static final String BLUETOOTH_PBAP_CLIENT_SET_PRIORITY = "bluetoothPbapClientSetPriority";

        Commands() {
        }
    }

    public static enum BluetoothConnectionState {
        DISCONNECTED(0),
        CONNECTING(1),
        CONNECTED(2),
        DISCONNECTING(3);

        private final int mState;

        private BluetoothConnectionState(int state) {
            this.mState = state;
        }

        public int getState() {
            return this.mState;
        }
    }

    public static enum BluetoothProfile {
        HEADSET(1),
        A2DP(2),
        HID_HOST(4),
        PAN(5),
        PBAP(6),
        GATT(7),
        GATT_SERVER(8),
        MAP(9),
        SAP(10),
        A2DP_SINK(11),
        AVRCP_CONTROLLER(12),
        HEADSET_CLIENT(16),
        PBAP_CLIENT(17),
        MAP_CLIENT(18);

        private static final Map<Integer, BluetoothProfile> sProfileToValue;
        private final int mProfile;

        public static BluetoothProfile valueOfProfile(int profile) {
            return sProfileToValue.get(profile);
        }

        private BluetoothProfile(int profile) {
            this.mProfile = profile;
        }

        public int getProfile() {
            return this.mProfile;
        }

        static {
            sProfileToValue = Stream.of(BluetoothProfile.values()).collect(Collectors.toMap(BluetoothProfile::getProfile, value -> value));
        }
    }

    public static enum BluetoothAccessLevel {
        ACCESS_UNKNOWN(0),
        ACCESS_ALLOWED(1),
        ACCESS_REJECTED(2);

        private final int mAccess;

        private BluetoothAccessLevel(int access) {
            this.mAccess = access;
        }

        public int getAccess() {
            return this.mAccess;
        }
    }

    public static enum BluetoothPriorityLevel {
        PRIORITY_AUTO_CONNECT(1000),
        PRIORITY_ON(100),
        PRIORITY_OFF(0),
        PRIORITY_UNDEFINED(-1);

        private final int mPriority;

        private BluetoothPriorityLevel(int priority) {
            this.mPriority = priority;
        }

        public int getPriority() {
            return this.mPriority;
        }
    }

    @VisibleForTesting
    static class Events {
        static final String BLUETOOTH_STATE_CHANGED_ON = "BluetoothStateChangedOn";
        static final String BLUETOOTH_STATE_CHANGED_OFF = "BluetoothStateChangedOff";
        static final String BLUETOOTH_PROFILE_CONNECTION_STATE_CHANGED = "BluetoothProfileConnectionStateChanged";

        Events() {
        }
    }
}

