/*
 * Decompiled with CFR 0.152.
 */
package com.android.tradefed.util.image;

import com.android.tradefed.build.IBuildInfo;
import com.android.tradefed.build.IDeviceBuildInfo;
import com.android.tradefed.device.DeviceNotAvailableException;
import com.android.tradefed.device.ITestDevice;
import com.android.tradefed.invoker.logger.InvocationMetricLogger;
import com.android.tradefed.invoker.tracing.CloseableTraceScope;
import com.android.tradefed.log.LogUtil;
import com.android.tradefed.result.error.ErrorIdentifier;
import com.android.tradefed.result.error.InfraErrorIdentifier;
import com.android.tradefed.targetprep.TargetSetupError;
import com.android.tradefed.util.CommandResult;
import com.android.tradefed.util.CommandStatus;
import com.android.tradefed.util.FileUtil;
import com.android.tradefed.util.RunUtil;
import com.android.tradefed.util.ZipUtil2;
import com.android.tradefed.util.executor.ParallelDeviceExecutor;
import com.android.tradefed.util.image.DeviceImageTracker;
import com.google.common.collect.ImmutableSet;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.junit.Assert;

public class IncrementalImageUtil {
    public static final Set<String> DYNAMIC_PARTITIONS_TO_DIFF = ImmutableSet.of("product.img", "system.img", "system_dlkm.img", "system_ext.img", "vendor.img", "vendor_dlkm.img", new String[0]);
    private final File mSrcImage;
    private final File mSrcBootloader;
    private final File mSrcBaseband;
    private final File mTargetImage;
    private final ITestDevice mDevice;
    private final File mCreateSnapshotBinary;
    private boolean mBootloaderNeedsRevert = false;
    private boolean mBasebandNeedsRevert = false;
    private File mSourceDirectory;
    private ParallelPreparation mParallelSetup;

    public static IncrementalImageUtil initialize(ITestDevice device, IDeviceBuildInfo build, File createSnapshot, boolean isIsolatedSetup) throws DeviceNotAvailableException {
        if (isIsolatedSetup) {
            LogUtil.CLog.d("test is configured with isolation grade, doesn't support incremental yet.");
            return null;
        }
        DeviceImageTracker.FileCacheTracker tracker = DeviceImageTracker.getDefaultCache().getBaselineDeviceImage(device.getSerialNumber());
        if (tracker == null) {
            LogUtil.CLog.d("Not tracking current baseline image.");
            return null;
        }
        if (!tracker.buildId.equals(device.getBuildId())) {
            LogUtil.CLog.d("On-device build isn't matching the cache.");
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.DEVICE_IMAGE_CACHE_MISMATCH, 1L);
            return null;
        }
        if (!tracker.branch.equals(build.getBuildBranch())) {
            LogUtil.CLog.d("Newer build is not on the same branch.");
            return null;
        }
        if (!tracker.flavor.equals(build.getBuildFlavor())) {
            LogUtil.CLog.d("Newer build is not on the build flavor.");
            return null;
        }
        if (!IncrementalImageUtil.isSnapshotSupported(device)) {
            LogUtil.CLog.d("Incremental flashing not supported.");
            return null;
        }
        String splTarget = IncrementalImageUtil.getSplVersion(build);
        String splBaseline = device.getProperty("ro.build.version.security_patch");
        if (splTarget != null && !splBaseline.equals(splTarget)) {
            LogUtil.CLog.d("Target SPL is '%s', while baseline is '%s", splTarget, splBaseline);
            return null;
        }
        File deviceImage = null;
        File bootloader = null;
        File baseband = null;
        try {
            deviceImage = IncrementalImageUtil.copyImage(tracker.zippedDeviceImage);
            bootloader = IncrementalImageUtil.copyImage(tracker.zippedBootloaderImage);
            baseband = IncrementalImageUtil.copyImage(tracker.zippedBasebandImage);
        }
        catch (IOException e) {
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.DEVICE_IMAGE_CACHE_MISMATCH, 1L);
            LogUtil.CLog.e(e);
            FileUtil.deleteFile(deviceImage);
            FileUtil.deleteFile(bootloader);
            FileUtil.deleteFile(baseband);
            return null;
        }
        InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.DEVICE_IMAGE_CACHE_ORIGIN, String.format("%s:%s:%s", tracker.branch, tracker.buildId, tracker.flavor));
        return new IncrementalImageUtil(device, deviceImage, bootloader, baseband, build.getDeviceImageFile(), createSnapshot);
    }

    public IncrementalImageUtil(ITestDevice device, File deviceImage, File bootloader, File baseband, File targetImage, File createSnapshot) {
        this.mDevice = device;
        this.mSrcImage = deviceImage;
        this.mSrcBootloader = bootloader;
        this.mSrcBaseband = baseband;
        this.mTargetImage = targetImage;
        if (createSnapshot != null) {
            File snapshot = null;
            try {
                File destDir = ZipUtil2.extractZipToTemp(createSnapshot, "create_snapshot");
                snapshot = FileUtil.findFile(destDir, "create_snapshot");
                FileUtil.chmodGroupRWX(snapshot);
            }
            catch (IOException e) {
                LogUtil.CLog.e(e);
            }
            this.mCreateSnapshotBinary = snapshot;
        } else {
            this.mCreateSnapshotBinary = null;
        }
        this.mParallelSetup = new ParallelPreparation(Thread.currentThread().getThreadGroup(), this.mSrcImage, this.mTargetImage);
        this.mParallelSetup.start();
    }

    private static File copyImage(File originalImage) throws IOException {
        File copy = FileUtil.createTempFile(FileUtil.getBaseName(originalImage.getName()), ".img");
        copy.delete();
        FileUtil.hardlinkFile(originalImage, copy);
        return copy;
    }

    public static boolean isSnapshotSupported(ITestDevice device) throws DeviceNotAvailableException {
        CommandResult whichOutput = device.executeShellV2Command("which snapshotctl");
        LogUtil.CLog.d("stdout: %s, stderr: %s", whichOutput.getStdout(), whichOutput.getStderr());
        if (!whichOutput.getStdout().contains("/system/bin/snapshotctl")) {
            return false;
        }
        CommandResult helpOutput = device.executeShellV2Command("snapshotctl");
        LogUtil.CLog.d("stdout: %s, stderr: %s", helpOutput.getStdout(), helpOutput.getStderr());
        return helpOutput.getStdout().contains("map-snapshots") || helpOutput.getStderr().contains("map-snapshots");
    }

    public void notifyBootloaderNeedsRevert() {
        this.mBootloaderNeedsRevert = true;
    }

    public void notifyBasebadNeedsRevert() {
        this.mBasebandNeedsRevert = true;
    }

    public static boolean isSnapshotInUse(ITestDevice device) throws DeviceNotAvailableException {
        CommandResult dumpOutput = device.executeShellV2Command("snapshotctl dump");
        LogUtil.CLog.d("stdout: %s, stderr: %s", dumpOutput.getStdout(), dumpOutput.getStderr());
        return !dumpOutput.getStdout().contains("Using snapuserd: 0");
    }

    public void updateDevice() throws DeviceNotAvailableException, TargetSetupError {
        try {
            this.internalUpdateDevice();
        }
        catch (DeviceNotAvailableException | TargetSetupError | RuntimeException e) {
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.INCREMENTAL_FLASHING_UPDATE_FAILURE, 1L);
            throw e;
        }
    }

    private void internalUpdateDevice() throws DeviceNotAvailableException, TargetSetupError {
        InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.INCREMENTAL_FLASHING_ATTEMPT_COUNT, 1L);
        if (this.mDevice.isStateBootloaderOrFastbootd()) {
            this.mDevice.rebootUntilOnline();
        }
        if (!this.mDevice.enableAdbRoot()) {
            throw new TargetSetupError("Failed to obtain root, this is required for incremental update.", (ErrorIdentifier)InfraErrorIdentifier.INCREMENTAL_FLASHING_ERROR);
        }
        try {
            this.mParallelSetup.join();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        if (this.mParallelSetup.getError() != null) {
            throw this.mParallelSetup.getError();
        }
        File srcDirectory = this.mParallelSetup.getSrcDirectory();
        File targetDirectory = this.mParallelSetup.getTargetDirectory();
        File workDir = this.mParallelSetup.getWorkDir();
        try (CloseableTraceScope ignored = new CloseableTraceScope("update_device");){
            this.logTargetInformation(targetDirectory);
            this.logPatchesInformation(workDir);
            this.mDevice.executeShellV2Command("mkdir -p /data/ndb");
            this.mDevice.executeShellV2Command("rm -rf /data/ndb/*.patch");
            this.mDevice.executeShellV2Command("snapshotctl unmap-snapshots");
            this.mDevice.executeShellV2Command("snapshotctl delete-snapshots");
            ArrayList pushTasks = new ArrayList();
            for (File f : workDir.listFiles()) {
                try (CloseableTraceScope push = new CloseableTraceScope("push:" + f.getName());){
                    pushTasks.add(() -> {
                        boolean success = f.isDirectory() ? this.mDevice.pushDir(f, "/data/ndb/") : this.mDevice.pushFile(f, "/data/ndb/" + f.getName());
                        LogUtil.CLog.d("Push status: %s. %s->%s", success, f, "/data/ndb/" + f.getName());
                        Assert.assertTrue(success);
                        return true;
                    });
                }
            }
            ParallelDeviceExecutor pushExec = new ParallelDeviceExecutor(pushTasks.size());
            pushExec.invokeAll(pushTasks, 0L, TimeUnit.MINUTES);
            if (pushExec.hasErrors()) {
                throw new RuntimeException(pushExec.getErrors().get(0));
            }
            CommandResult mapOutput = this.mDevice.executeShellV2Command("snapshotctl map-snapshots /data/ndb/");
            LogUtil.CLog.d("stdout: %s, stderr: %s", mapOutput.getStdout(), mapOutput.getStderr());
            if (!CommandStatus.SUCCESS.equals((Object)mapOutput.getStatus())) {
                throw new TargetSetupError(String.format("Failed to map the snapshots.\nstdout:%s\nstderr:%s", mapOutput.getStdout(), mapOutput.getStderr()), (ErrorIdentifier)InfraErrorIdentifier.INCREMENTAL_FLASHING_ERROR);
            }
            this.flashStaticPartition(targetDirectory);
            this.mSourceDirectory = srcDirectory;
            this.mDevice.enableAdbRoot();
            CommandResult psOutput = this.mDevice.executeShellV2Command("ps -ef | grep snapuserd");
            LogUtil.CLog.d("stdout: %s, stderr: %s", psOutput.getStdout(), psOutput.getStderr());
        }
        catch (DeviceNotAvailableException | RuntimeException e) {
            if (this.mSourceDirectory == null) {
                FileUtil.recursiveDelete(srcDirectory);
            }
            throw e;
        }
        finally {
            FileUtil.recursiveDelete(workDir);
            FileUtil.recursiveDelete(targetDirectory);
        }
    }

    public void teardownDevice() throws DeviceNotAvailableException {
        try (CloseableTraceScope ignored = new CloseableTraceScope("teardownDevice");){
            if (this.mBootloaderNeedsRevert) {
                this.mDevice.rebootIntoBootloader();
                CommandResult bootloaderFlashTarget = this.mDevice.executeFastbootCommand("flash", "bootloader", this.mSrcBootloader.getAbsolutePath());
                LogUtil.CLog.d("Status: %s", new Object[]{bootloaderFlashTarget.getStatus()});
                LogUtil.CLog.d("stdout: %s", bootloaderFlashTarget.getStdout());
                LogUtil.CLog.d("stderr: %s", bootloaderFlashTarget.getStderr());
            }
            if (this.mBasebandNeedsRevert) {
                this.mDevice.rebootIntoBootloader();
                CommandResult radioFlashTarget = this.mDevice.executeFastbootCommand("flash", "radio", this.mSrcBaseband.getAbsolutePath());
                LogUtil.CLog.d("Status: %s", new Object[]{radioFlashTarget.getStatus()});
                LogUtil.CLog.d("stdout: %s", radioFlashTarget.getStdout());
                LogUtil.CLog.d("stderr: %s", radioFlashTarget.getStderr());
            }
            if (this.mDevice.isStateBootloaderOrFastbootd()) {
                this.mDevice.reboot();
            }
            this.mDevice.enableAdbRoot();
            CommandResult revertOutput = this.mDevice.executeShellV2Command("snapshotctl revert-snapshots");
            if (!CommandStatus.SUCCESS.equals((Object)revertOutput.getStatus())) {
                LogUtil.CLog.d("Failed revert-snapshots. stdout: %s, stderr: %s", revertOutput.getStdout(), revertOutput.getStderr());
                InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.INCREMENTAL_FLASHING_TEARDOWN_FAILURE, 1L);
            }
            if (this.mSourceDirectory != null) {
                this.flashStaticPartition(this.mSourceDirectory);
            }
        }
        catch (DeviceNotAvailableException e) {
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationMetricKey.INCREMENTAL_FLASHING_TEARDOWN_FAILURE, 1L);
            throw e;
        }
        finally {
            FileUtil.recursiveDelete(this.mSourceDirectory);
            FileUtil.deleteFile(this.mSrcImage);
            FileUtil.deleteFile(this.mSrcBootloader);
            FileUtil.deleteFile(this.mSrcBaseband);
        }
    }

    private void blockCompare(File srcImage, File targetImage, File workDir) {
        try (CloseableTraceScope ignored = new CloseableTraceScope("block_compare:" + srcImage.getName());){
            CommandResult result;
            RunUtil runUtil = new RunUtil();
            runUtil.setWorkingDir(workDir);
            String createSnapshot = "create_snapshot";
            if (this.mCreateSnapshotBinary != null && this.mCreateSnapshotBinary.exists()) {
                createSnapshot = this.mCreateSnapshotBinary.getAbsolutePath();
            }
            if (!CommandStatus.SUCCESS.equals((Object)(result = runUtil.runTimedCmd(0L, createSnapshot, "--source=" + srcImage.getAbsolutePath(), "--target=" + targetImage.getAbsolutePath())).getStatus())) {
                throw new RuntimeException(String.format("%s\n%s", result.getStdout(), result.getStderr()));
            }
            File[] listFiles = workDir.listFiles();
            LogUtil.CLog.d("%s", Arrays.asList(listFiles));
        }
    }

    private boolean flashStaticPartition(File imageDirectory) throws DeviceNotAvailableException {
        this.mDevice.rebootIntoBootloader();
        HashMap<String, String> envMap = new HashMap<String, String>();
        envMap.put("ANDROID_PRODUCT_OUT", imageDirectory.getAbsolutePath());
        CommandResult fastbootResult = this.mDevice.executeLongFastbootCommand(envMap, "flashall", "--exclude-dynamic-partitions", "--disable-super-optimization");
        LogUtil.CLog.d("Status: %s", new Object[]{fastbootResult.getStatus()});
        LogUtil.CLog.d("stdout: %s", fastbootResult.getStdout());
        LogUtil.CLog.d("stderr: %s", fastbootResult.getStderr());
        if (!CommandStatus.SUCCESS.equals((Object)fastbootResult.getStatus())) {
            return false;
        }
        this.mDevice.waitForDeviceAvailable(300000L);
        return true;
    }

    private void logPatchesInformation(File patchesDirectory) {
        for (File patch : patchesDirectory.listFiles()) {
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationGroupMetricKey.INCREMENTAL_FLASHING_PATCHES_SIZE, patch.getName(), patch.length());
        }
    }

    private void logTargetInformation(File targetDirectory) {
        for (File patch : targetDirectory.listFiles()) {
            if (!DYNAMIC_PARTITIONS_TO_DIFF.contains(patch.getName())) continue;
            InvocationMetricLogger.addInvocationMetrics(InvocationMetricLogger.InvocationGroupMetricKey.INCREMENTAL_FLASHING_TARGET_SIZE, patch.getName(), patch.length());
        }
    }

    private static String getSplVersion(IBuildInfo build) {
        File buildProp = build.getFile("build.prop");
        if (buildProp == null) {
            LogUtil.CLog.d("No target build.prop found for comparison.");
            return null;
        }
        try {
            String props = FileUtil.readStringFromFile(buildProp);
            for (String line : props.split("\n")) {
                if (!line.startsWith("ro.build.version.security_patch=")) continue;
                return line.split("=")[1];
            }
        }
        catch (IOException e) {
            LogUtil.CLog.e(e);
        }
        return null;
    }

    private class ParallelPreparation
    extends Thread {
        private final File mSetupSrcImage;
        private final File mSetupTargetImage;
        private File mSrcDirectory;
        private File mTargetDirectory;
        private File mWorkDir;
        private TargetSetupError mError;

        public ParallelPreparation(ThreadGroup currentGroup, File srcImage, File targetImage) {
            super(currentGroup, "incremental-flashing-preparation");
            this.setDaemon(true);
            this.mSetupSrcImage = srcImage;
            this.mSetupTargetImage = targetImage;
        }

        @Override
        public void run() {
            try (CloseableTraceScope ignored = new CloseableTraceScope("unzip_device_images");){
                String[] futureSrcDir = CompletableFuture.supplyAsync(() -> {
                    try {
                        return ZipUtil2.extractZipToTemp(this.mSetupSrcImage, "incremental_src");
                    }
                    catch (IOException ioe) {
                        throw new RuntimeException(ioe);
                    }
                });
                CompletableFuture<File> futureTargetDir = CompletableFuture.supplyAsync(() -> {
                    try {
                        return ZipUtil2.extractZipToTemp(this.mSetupTargetImage, "incremental_target");
                    }
                    catch (IOException ioe) {
                        throw new RuntimeException(ioe);
                    }
                });
                this.mSrcDirectory = (File)futureSrcDir.get();
                this.mTargetDirectory = (File)futureTargetDir.get();
            }
            catch (InterruptedException | ExecutionException e) {
                FileUtil.recursiveDelete(this.mSrcDirectory);
                FileUtil.recursiveDelete(this.mTargetDirectory);
                this.mSrcDirectory = null;
                this.mTargetDirectory = null;
                this.mError = new TargetSetupError(e.getMessage(), (Throwable)e, (ErrorIdentifier)InfraErrorIdentifier.FAIL_TO_CREATE_FILE);
                return;
            }
            try {
                this.mWorkDir = FileUtil.createTempDir("block_compare_workdir");
            }
            catch (IOException e) {
                FileUtil.recursiveDelete(this.mWorkDir);
                FileUtil.recursiveDelete(this.mSrcDirectory);
                FileUtil.recursiveDelete(this.mTargetDirectory);
                this.mSrcDirectory = null;
                this.mTargetDirectory = null;
                this.mError = new TargetSetupError(e.getMessage(), (Throwable)e, (ErrorIdentifier)InfraErrorIdentifier.FAIL_TO_CREATE_FILE);
                return;
            }
            ArrayList callableTasks = new ArrayList();
            for (String partition : this.mSrcDirectory.list()) {
                File possibleSrc = new File(this.mSrcDirectory, partition);
                File possibleTarget = new File(this.mTargetDirectory, partition);
                File workDirectory = this.mWorkDir;
                if (possibleSrc.exists() && possibleTarget.exists()) {
                    if (!DYNAMIC_PARTITIONS_TO_DIFF.contains(partition)) continue;
                    callableTasks.add(() -> {
                        IncrementalImageUtil.this.blockCompare(possibleSrc, possibleTarget, workDirectory);
                        return true;
                    });
                    continue;
                }
                LogUtil.CLog.e("Skipping %s no src or target", partition);
            }
            ParallelDeviceExecutor executor = new ParallelDeviceExecutor(callableTasks.size());
            executor.invokeAll(callableTasks, 0L, TimeUnit.MINUTES);
            if (executor.hasErrors()) {
                this.mError = new TargetSetupError(executor.getErrors().get(0).getMessage(), executor.getErrors().get(0), (ErrorIdentifier)InfraErrorIdentifier.BLOCK_COMPARE_ERROR);
            }
        }

        public File getSrcDirectory() {
            return this.mSrcDirectory;
        }

        public File getTargetDirectory() {
            return this.mTargetDirectory;
        }

        public File getWorkDir() {
            return this.mWorkDir;
        }

        public TargetSetupError getError() {
            return this.mError;
        }
    }
}

